import asyncio
import datetime
from logging import getLogger
from typing import Callable

from async_tools import acall


logger = getLogger(__name__)


class Periodic:
    def __init__(
            self,
            job: Callable,
            period: datetime.timedelta,
            is_active: bool = True,
            first_at: datetime.datetime = None,
            decrease_sleep_time_by_evaluation_time: bool = False,
    ) -> None:
        self.job = job
        self.period = period
        self.is_active = is_active
        self.first_at = first_at
        self.decrease_sleep_time_by_evaluation_time = decrease_sleep_time_by_evaluation_time
        self.future = None
        if is_active:
            self.start()

    def __str__(self):
        period = repr(self.period).removeprefix("datetime.")
        return f"Periodic(job={self.job.__name__}, period={period})"

    def start(self):
        if self.future is not None:
            logger.warning(f"start called on already running {self}, ignored")
            return
        logger.info(f"{self} started")
        self.future = asyncio.ensure_future(self.loop())

    async def loop(self):
        if self.first_at:
            sleep_time = (self.first_at - datetime.datetime.now()).total_seconds()
            if sleep_time > 0:
                await asyncio.sleep(sleep_time)
            else:
                logger.warning(f"{self} got first_at before now, skip sleep")
        else:
            await asyncio.sleep(self.period.total_seconds())

        while self.is_active:
            logger.info(f"{self} starting job: {self.job}")
            start_at = datetime.datetime.now()
            try:
                await acall(self.job)
            except Exception as e:
                logger.error(f"{self} job raised an Exception: {repr(e)}")
                logger.exception(e)
            logger.debug(f"{self} finished job: {self.job}")
            if not self.is_active:
                break
            sleep_time = self.period.total_seconds()
            if self.decrease_sleep_time_by_evaluation_time:
                sleep_time -= (datetime.datetime.now() - start_at).total_seconds()
            if sleep_time > 0:
                logger.debug(f"{self} sleep {sleep_time} sec till next job call "
                             f"(at {datetime.datetime.now() + datetime.timedelta(seconds=sleep_time)})")
                await asyncio.sleep(sleep_time)
            else:
                logger.warning(f"{self} has not time to sleep, initiate next loop")
        logger.warning(f"{self} RETURNED")
