#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mike Orlov <m.orlov@abm-jsc.ru>
#
import asyncio
import itertools
import typing
from logging import getLogger

logger = getLogger(__name__)


class AsyncDeinitable:
    def __init__(self):
        self.async_deinit_scheduled__event: asyncio.Event = asyncio.Event()  # called async_deinit
        self._tasks_before_async_deinit: list[asyncio.Task] = []
        self.async_deinit_started__event: asyncio.Event = asyncio.Event()  # passed wait
        self.async_deinit_finished__event: asyncio.Event = asyncio.Event()  # done own _async_deinit

    def wait_before_async_deinit_for(self, task: asyncio.Task) -> None:
        self._tasks_before_async_deinit.append(task)

    @typing.final
    async def async_deinit(self) -> None:
        if self.async_deinit_scheduled__event.is_set():
            logger.warning(f"async_start called extra time on {self}")
            return
        self.async_deinit_scheduled__event.set()
        if self._tasks_before_async_deinit:
            logger.debug(f"{self} before starting own _async_deinit is waiting for {self._tasks_before_async_deinit}")
            await asyncio.gather(*self._tasks_before_async_deinit)
        self.async_deinit_started__event.set()
        logger.debug(f"{self} started _async_deinit")
        await self._async_deinit()
        logger.debug(f"{self} finished _async_deinit")
        self.async_deinit_finished__event.set()

    async def _async_deinit(self) -> None:
        """Если дефолтная деинициализация не подходит, то переопределите этот метод"""
        await self._async_deinit_attrs()

    async def _async_deinit_attrs(self) -> None:
        async with asyncio.TaskGroup() as tg:
            for attribute_name in vars(self):
                attribute = getattr(self, attribute_name)
                if isinstance(attribute, AsyncDeinitable) and not attribute.async_deinit_scheduled__event.is_set():
                    tg.create_task(attribute.async_deinit())
                if isinstance(attribute, (list, set)):
                    for val in attribute:
                        if isinstance(val, AsyncDeinitable) and not val.async_deinit_scheduled__event.is_set():
                            tg.create_task(val.async_deinit())
                if isinstance(attribute, dict):
                    for val in itertools.chain(attribute.keys(), attribute.values()):
                        if isinstance(val, AsyncDeinitable) and not val.async_deinit_scheduled__event.is_set():
                            tg.create_task(val.async_deinit())
