#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors:
#  Vasya Svintsov <v.svintsov@techokert.ru>,
#  Alexander Medvedev <a.medvedev@abm-jsc.ru>,
#  Andrey Vaydich <a.vaydich@abm-jsc.ru>

from dataclasses import dataclass
from typing import Optional

import furl
from aiohttp.hdrs import METH_HEAD, RANGE
from aiohttp.web_exceptions import HTTPUnprocessableEntity
from dict_caster import Item, DictCaster
from http_tools import HttpServer, Answer, FileAnswer, WrappedAnswer, HttpStatusCode
from http_tools.request import IncomingRequest

from .file_controller import FileController
from .file_descriptor import get_file_descriptor_from_file_metadata
from .range import Range
from .tools.error_middleware import error_middleware
from .tools.get_file_headers import get_file_headers


class FileHandler:
    @dataclass
    class Context:
        http_server: HttpServer
        file_controller: FileController

    def __init__(self, context: Context, path_prefix: Optional[str] = None) -> None:
        self._context = context

        self._context.http_server.register_handler(str(furl.Path(path_prefix).add('/file/add')), self.add)
        self._context.http_server.register_handler(str(furl.Path(path_prefix).add('/file/get')), self.get)
        self._context.http_server.register_handler(
            str(furl.Path(path_prefix).add('/file/zip')), self.zip_files
        )
        self._context.http_server.register_handler(str(furl.Path(path_prefix).add('/file/info')), self.info)
        self._context.http_server.register_handler(str(furl.Path(path_prefix).add('/file/delete')), self.delete)

    @error_middleware
    async def add(self, request: IncomingRequest) -> Answer:
        if file := request.key_value_arguments.get('file'):  # multipart/form-data
            file_name = file.filename
            content = file.content
        elif request.payload:  # application/octet-stream
            file_name = DictCaster(Item('filename', str)).cast_and_return(request.key_value_arguments)
            content = request.payload
        else:
            raise HTTPUnprocessableEntity(reason='FileMissing')

        is_create_thumbnail = DictCaster(
            Item('is_create_thumbnail', bool, default=False),
        ).cast_and_return(request.key_value_arguments)

        file_metadata = await self._context.file_controller.upload_file(file_name, content, is_create_thumbnail)
        file_descriptor = get_file_descriptor_from_file_metadata(file_metadata)
        return WrappedAnswer(file_descriptor)

    @error_middleware
    async def get(self, request: IncomingRequest) -> Answer:
        storage_file_name = DictCaster(
            Item('id', str)
        ).cast_and_return(request.key_value_arguments)

        if request.metadata.request_method == METH_HEAD:
            return await self._get_file_metadata(storage_file_name)

        range_ = None
        if range_header := request.metadata.get_header(RANGE):
            range_ = Range.from_value(range_header)

        return await self._get_file(storage_file_name, range_)

    async def _get_file_metadata(self, storage_file_name: str) -> Answer:
        file_metadata = await self._context.file_controller.get_file_info(storage_file_name)
        file_descriptor = get_file_descriptor_from_file_metadata(file_metadata)
        return Answer(payload=None,
                      status=HttpStatusCode.OK,
                      content_type=file_descriptor.content_type,
                      headers=get_file_headers(file_descriptor)
                      )

    async def _get_file(self, storage_file_name: str, range_: Range | None = None) -> Answer:
        file_metadata, file_payload = await self._context.file_controller.get_file(storage_file_name, range_)
        file_descriptor = get_file_descriptor_from_file_metadata(file_metadata, file_payload)
        return FileAnswer(payload=file_descriptor.bytes_content,
                          file_name=file_descriptor.quoted_full_file_name,
                          content_type=file_descriptor.content_type,
                          )

    @error_middleware
    async def zip_files(self, request: IncomingRequest) -> Answer:
        storage_file_names = DictCaster(
            Item('storage_file_names', list[str])
        ).cast_and_return(request.key_value_arguments)

        # Todo: HEAD + RANGE ??

        file_metadata, file_payload = await self._context.file_controller.zip_files(storage_file_names)
        # Todo: Need to patch get_file_descriptor_from_file_metadata method
        #  for correct content_type typing (now it's str)
        file_descriptor = get_file_descriptor_from_file_metadata(file_metadata, file_payload)

        return FileAnswer(payload=file_descriptor.bytes_content,
                          file_name=file_descriptor.quoted_full_file_name,
                          content_type=file_descriptor.content_type)

    @error_middleware
    async def info(self, request: IncomingRequest) -> Answer:
        storage_file_name = DictCaster(
            Item('id', str)
        ).cast_and_return(request.key_value_arguments)

        file_metadata = await self._context.file_controller.get_file_info(storage_file_name)
        file_descriptor = get_file_descriptor_from_file_metadata(file_metadata)
        return WrappedAnswer(file_descriptor)

    @error_middleware
    async def delete(self, request: IncomingRequest) -> Answer:
        storage_file_name = DictCaster(
            Item('id', str)
        ).cast_and_return(request.key_value_arguments)

        file_metadata = await self._context.file_controller.delete_file(storage_file_name)
        file_descriptor = get_file_descriptor_from_file_metadata(file_metadata)
        return WrappedAnswer(file_descriptor)
