#  Copyright (C) 2024
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mike Orlov <m.orlov@abm-jsc.ru>
import logging
from dataclasses import dataclass
from typing import Type

from file_storage.abstract_file_storage import AbstractFileStorage

from .entities.thumbnail_info import ThumbnailInfo
from .database import Database, WrappedSession
from .entities.link_info import LinkInfo
from .extra import silent, AnySession
from .file_keeper import FileKeeper
from .link_keeper import LinkKeeper, LI
from .memory_zipper import MemoryZipper
from .thumbnail.thumbnail_provider import ThumbnailProvider

logger = logging.getLogger(__name__)


class Controller:
    @dataclass
    class Config:
        pass

    @dataclass
    class Context:
        database: Database
        # zipper: MemoryZipper
        file_storage: AbstractFileStorage
        # image_thumbnail_processor: ImageThumbnailProcessor
        thumbnail_provider: ThumbnailProvider
        # file_database_facade: FileDatabaseFacade
        # thumbnail_provider: ThumbnailProvider

    def __init__(self,
                 config: Config, context: Context,
                 link_class: Type[LI] = LinkInfo, thumbnail_class: Type[LI] = ThumbnailInfo,
                 ) -> None:
        self.config = config
        self.context = context

        self.file_keeper = FileKeeper(FileKeeper.Context(self.context.database, self.context.file_storage))
        self.link_keeper = LinkKeeper(
            LinkKeeper.Context(self.file_keeper, self.context.database), link_class=link_class
        )
        self.thumbnail_keeper = LinkKeeper(
            LinkKeeper.Context(self.file_keeper, self.context.database), link_class=thumbnail_class
        )
        self.zipper = MemoryZipper(MemoryZipper.Context(self.link_keeper, self.context.database))

    def _publish(self):
        ...

    async def _find_thumbnail_key_by_file_key(self, file_info_key: str, session:  WrappedSession) -> str | None:
        if link_infos_with_same_key := await self.link_keeper.get_link_infos_by_file_info_key(file_info_key, session):
            links_ids = [link.id for link in link_infos_with_same_key]
            if thumbnail_infos := await self.thumbnail_keeper.get_link_infos(links_ids, session, limit=1):
                return thumbnail_infos[0].file_info_key

    async def upload(
            self, content: bytes, filename: str, new_link_id: str | None = None, session: AnySession = None,
            is_thumbnail_required: bool = False) -> tuple[LinkInfo, ThumbnailInfo | None]:
        thumbnail_info = None
        async with self.context.database.ensure_session(session) as session:
            link_info = await self.link_keeper.upload(content, filename, link_id=new_link_id, session=session)
            if is_thumbnail_required:
                if thumbnail_key := await self._find_thumbnail_key_by_file_key(link_info.file_info_key, session):
                    logger.info("Found existing thumbnail, using it")
                    thumbnail_info = await self.thumbnail_keeper.add(thumbnail_key, filename, link_info.id, session)
                elif thumbnail := await self.context.thumbnail_provider.provide(link_info.extension, content):
                    logger.info("Existing thumbnail not found, created new")
                    thumbnail_info = await self.thumbnail_keeper.upload(thumbnail, filename, link_info.id, session)
        return link_info, thumbnail_info

    async def get(self, link_id: str, byte_range: range | None = None,
                  session: AnySession = None) -> tuple[LinkInfo, bytes]:
        link_info, file_content = await self.link_keeper.get(link_id, byte_range=byte_range, session=session)
        return link_info, file_content

    async def get_thumbnail(self, link_id: str, byte_range: range | None = None,
                            session: AnySession = None) -> tuple[LinkInfo, bytes]:
        link_info, file_content = await self.thumbnail_keeper.get(link_id, byte_range=byte_range, session=session)
        return link_info, file_content

    async def read(self, file_info_key: str, byte_range: range | None = None) -> bytes:
        return await self.link_keeper.read(file_info_key, byte_range)

    async def head(self, link_id: str, session: AnySession = None) -> LinkInfo:
        return await self.link_keeper.head(link_id, session=session)

    async def delete(self, link_id: str, session: AnySession = None) -> LinkInfo:
        async with self.context.database.ensure_session(session) as session:
            result = await self.link_keeper.delete(link_id, session)
            await silent(self.thumbnail_keeper.delete(link_id, session))
        return result

    async def zip_files(self, link_ids: list[str], session: AnySession = None) -> tuple[LinkInfo, bytes]:
        return await self.zipper.zip_files(link_ids, session=session)
