#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>

from abc import ABC, abstractmethod
from dataclasses import dataclass
from typing import Any, Literal, Awaitable

from prometheus_tools.metrics_registry import MetricsRegistry


class AbstractFileStorage(ABC):
    @dataclass
    class Config:
        root_dir: str = ""
        max_capacity_gb: float = 1.0

        def __post_init__(self) -> None:
            self.root_dir = self.root_dir.lstrip("/")

        @property
        def max_capacity_b(self) -> float:
            return self.max_capacity_gb * (1024 ** 3)

        @property
        def max_capacity_kb(self) -> float:
            return self.max_capacity_gb * (1024 ** 2)

        @property
        def max_capacity_mb(self) -> float:
            return self.max_capacity_gb * 1024

    @dataclass
    class Context:
        metrics: MetricsRegistry

    def __init__(self, config: Config, context: Context) -> None:
        self.config = config
        self.context = context
        self.class_name = type(self).__name__

    async def save(self, key: str, value: bytes, allow_rewrite: bool = False) -> None:
        return await self._wrap_execution('save', self._save(key, value, allow_rewrite))

    async def load(self, key: str, offset: int = 0, size: int = -1) -> bytes:
        return await self._wrap_execution('load', self._load(key, offset, size))

    async def delete(self, key: str) -> None:
        return await self._wrap_execution('delete', self._delete(key))

    async def check_file_existence(self, key: str) -> None:
        return await self._wrap_execution('check_file_existence', self._check_file_existence(key))

    async def _wrap_execution(
            self, action: Literal['save', 'load', 'delete', 'check_file_existence'], handler: Awaitable[Any]
    ) -> Any:
        with self.context.metrics.track(
                'file_storage__handlers', {'storage': self.class_name, 'action': action}, except_labels={'error': None}
        ) as tracker:
            try:
                return await handler
            except Exception as er:
                tracker.labels['error'] = type(er).__name__
                raise

    def _split_key(self, key: str, division_index: int = 3) -> tuple[str, str]:
        if len(key) <= division_index:
            raise ValueError(f"Impossible to split '{key}' because the size is less or equal than {division_index}")
        return key[:division_index], key[division_index:]

    @abstractmethod
    async def _save(self, key: str, value: bytes, allow_rewrite: bool = False) -> None:
        pass

    @abstractmethod
    async def _load(self, key: str, offset: int = 0, size: int = -1) -> bytes:
        pass

    @abstractmethod
    async def _delete(self, key: str) -> None:
        pass

    @abstractmethod
    async def _check_file_existence(self, key: str) -> None:
        pass
