#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>

from abc import ABC, abstractmethod
from dataclasses import dataclass, field
from enum import StrEnum
from typing import Any, Awaitable

from prometheus_tools.metrics_registry import MetricsRegistry

from file_storage.exceptions import FileStorageFull


@dataclass
class StorageCapacity:
    max_b: float
    used_b: float

    def __post_init__(self):
        if self.max_b <= 0:
            raise ValueError("max_b must be > 0")

    @property
    def free_b(self) -> float:
        return max(self.max_b - self.used_b, 0.0)

    @property
    def used_percentage(self) -> float:
        return round(min(self.used_b / self.max_b * 100, 100), 2)

    def increase_used_space(self, content_size: int) -> None:
        self.used_b += content_size

    def decrease_used_space(self, content_size: int) -> None:
        self.used_b = max(self.used_b - content_size, 0)

    def check_free_space(self, content_size: int) -> None:
        if content_size > self.free_b:
            raise FileStorageFull(f'Not enough disk space. Free storage size {self.free_b} bytes, '
                                  f'but need {content_size} bytes')


class ActionExecution(StrEnum):
    SAVE = 'save'
    LOAD = 'load'
    DELETE = 'delete'
    CHECK_FILE_EXISTENCE = 'check_file_existence'


class AbstractFileStorage(ABC):
    @dataclass
    class Config:
        root_dir: str | None = None
        max_capacity_gb: float = field(kw_only=True, default=float("inf"))

        def __post_init__(self) -> None:
            self.root_dir = self.root_dir.lstrip("/") if self.root_dir else None

        @property
        def max_capacity_b(self) -> float:
            return self.max_capacity_gb * (1024 ** 3)

        @property
        def max_capacity_kb(self) -> float:
            return self.max_capacity_gb * (1024 ** 2)

        @property
        def max_capacity_mb(self) -> float:
            return self.max_capacity_gb * 1024

    @dataclass
    class Context:
        metrics: MetricsRegistry

    def __init__(self, config: Config, context: Context) -> None:
        self.config = config
        self.context = context
        self.class_name = type(self).__name__
        self.status: StorageCapacity | None = None

    async def save(self, key: str, value: bytes, allow_rewrite: bool = False) -> None:
        content_size = len(value)
        self.status.check_free_space(content_size)
        await self._wrap_execution(ActionExecution.SAVE, self._save(key, value, allow_rewrite))
        self.status.increase_used_space(content_size)

    async def load(self, key: str, offset: int = 0, size: int = -1) -> bytes:
        return await self._wrap_execution(ActionExecution.LOAD, self._load(key, offset, size))

    async def delete(self, key: str) -> None:
        content_size = await self._get_file_size(key)
        await self._wrap_execution(ActionExecution.DELETE, self._delete(key))
        self.status.decrease_used_space(content_size) if content_size else None

    async def check_file_existence(self, key: str) -> None:
        return await self._wrap_execution(ActionExecution.CHECK_FILE_EXISTENCE, self._check_file_existence(key))

    async def _wrap_execution(
            self, action: ActionExecution, handler: Awaitable[Any]
    ) -> Any:
        with self.context.metrics.track(
                'file_storage__handlers', {'storage': self.class_name, 'action': action}, except_labels={'error': None}
        ) as tracker:
            try:
                return await handler
            except Exception as er:
                tracker.labels['error'] = type(er).__name__
                raise

    @staticmethod
    def _split_key(key: str, division_index: int = 3) -> tuple[str, str]:
        if len(key) <= division_index:
            raise ValueError(f"Impossible to split '{key}' because the size is less or equal than {division_index}")
        return key[:division_index], key[division_index:]

    @abstractmethod
    async def _save(self, key: str, value: bytes, allow_rewrite: bool = False) -> None:
        pass

    @abstractmethod
    async def _load(self, key: str, offset: int = 0, size: int = -1) -> bytes:
        pass

    @abstractmethod
    async def _delete(self, key: str) -> None:
        pass

    @abstractmethod
    async def _check_file_existence(self, key: str) -> None:
        pass

    @abstractmethod
    async def _get_file_size(self, key: str) -> int:
        pass
