#  Copyright (C) 2024
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>

from dataclasses import dataclass
from typing import Optional

from http_tools import AbmServiceConnector
from init_helpers.dict_to_dataclass import get_converter


class NetrisProviderConnector:
    Config = AbmServiceConnector.Config

    @dataclass
    class Context(AbmServiceConnector.Context):
        project_name: str

    def __init__(self, config: Config, context: Context) -> None:
        self.config = config
        self.context = context
        self._connector = AbmServiceConnector(config, context)

    async def get_camera_outer_id_to_inner_id(self, outer_ids: list[str]) -> dict[str, int]:
        camera_outer_id_to_inner_id = await self._connector.post(
            '/camera_matching/outer_id_to_inner_id/get', {'ids': outer_ids}, headers=self._prepare_headers(),
        )
        return get_converter(dict[str, int])(camera_outer_id_to_inner_id)

    async def get_camera_inner_id_to_outer_id(self, inner_ids: list[int]) -> dict[int, str]:
        camera_inner_id_to_outer_id = await self._connector.post(
            '/camera_matching/inner_id_to_outer_id/get', {'ids': inner_ids}, headers=self._prepare_headers(),
        )
        return get_converter(dict[int, str])(camera_inner_id_to_outer_id)

    async def get_snapshot(self, uuid: Optional[str] = None, numeric_ext_id: Optional[int] = None) -> bytes:
        if (uuid and numeric_ext_id) or (not uuid and not numeric_ext_id):
            raise ValueError("required exactly one value of 'uuid' or 'numeric_ext_id'")

        payload = {}
        if uuid:
            payload['uuid'] = uuid
        if numeric_ext_id:
            payload['numeric_ext_id'] = numeric_ext_id

        snapshot = await self._connector.get(
            '/snapshot/get',
            url_query=payload,
            headers=self._prepare_headers()
        )

        return snapshot

    def _prepare_headers(self) -> dict[str, str]:
        return {"server_name": self.context.project_name}
