#  Copyright (C) 2022
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasiliev Ivan <i.vasiliev@technokert.ru>

import asyncio
import logging
from dataclasses import dataclass
from typing import Any, Optional, NoReturn

logger = logging.getLogger(__file__)


class AbstractBuffer:

    @dataclass
    class Config:
        MAX_VALUES_IN_BUFFER: int = 10000
        MAX_DELAY_BETWEEN_PROCESS: int = 5

    def __init__(self, handle_buffer, config: Config = Config()) -> None:
        self.__handle_buffer = handle_buffer
        self._config = config
        self._buffer = []
        self._new_value_event_event = asyncio.Event()
        self._buffer_flushed_event = asyncio.Event()
        self._process_buffer_lock = asyncio.Lock()
        self._lazy_process_task: Optional[asyncio.Task] = None

    async def __process_buffer_loop(self) -> NoReturn:
        while True:
            await self._new_value_event_event.wait()
            self._new_value_event_event.clear()
            async with self._process_buffer_lock:
                if len(self._buffer) >= self._config.MAX_VALUES_IN_BUFFER:
                    self._unschedule_lazy_proceed()
                    await self.__process_buffer()
                else:
                    self._schedule_lazy_proceed()

    def _schedule_lazy_proceed(self) -> None:
        if self._lazy_process_task is None or self._lazy_process_task.done():
            logger.debug("creating lazy process task ")
            self._lazy_process_task = asyncio.create_task(self._lazy_process())

    def _unschedule_lazy_proceed(self) -> None:
        if self._lazy_process_task is not None and not self._lazy_process_task.done():
            logger.debug("removing  lazy process task ")
            self._lazy_process_task.cancel()

    async def _lazy_process(self) -> None:
        await asyncio.sleep(self._config.MAX_DELAY_BETWEEN_PROCESS)
        logger.info(f" lazy process fired after {self._config.MAX_DELAY_BETWEEN_PROCESS} delay,"
                    f" buffer_size {len(self._buffer)}")
        async with self._process_buffer_lock:
            await self.__process_buffer()

    async def add_to_buffer(self, value: Any) -> None:
        if len(self._buffer) >= self._config.MAX_VALUES_IN_BUFFER:
            await self._buffer_flushed_event.wait()
        self._buffer.append(value)
        self._new_value_event_event.set()
        self._buffer_flushed_event.clear()

    async def __process_buffer(self) -> None:
        logger.debug("buffer processing started")
        values = self._buffer
        self._buffer = []
        try:
            if len(values) > self._config.MAX_VALUES_IN_BUFFER:
                for i in range(0, (len(values) // self._config.MAX_VALUES_IN_BUFFER) + 1):
                    part_to_handle = values[i*self._config.MAX_VALUES_IN_BUFFER:(i+1)*self._config.MAX_VALUES_IN_BUFFER]
                    if part_to_handle:
                        await self.__handle_buffer(part_to_handle)
            else:
                await self.__handle_buffer(values)
        except Exception as e:
            logger.warning(f"buffer processing failed with {values}; error: {repr(e)}")
        self._buffer_flushed_event.set()
        logger.debug("buffer processing finished")

    async def run(self) -> NoReturn:
        await self.__process_buffer_loop()
