import abc
import asyncio
import copy
import dataclasses
import functools
import itertools
import time
from dataclasses import dataclass, InitVar
from typing import Type, Iterable, Any, Optional, Union, Tuple, TypeVar, Callable, Awaitable, Mapping
from types import MappingProxyType

import sqlalchemy
import sqlalchemy.orm
from async_tools import acall
from dict_caster.extras import to_list
from dynamic_types.class_name import _prepare_class_name
from dynamic_types.create import create_type

from init_helpers.custom_json import ReprInDumps
from init_helpers.dict_to_dataclass import get_dataclass_field_name_to_field, NoValue, dict_to_dataclass
from sqlalchemy import Integer, Column

from extended_logger import get_logger
from ..entity_field import NoDefault


logger = get_logger(__name__)


def create_dataclass(new_type_name: str, bases: Iterable[type], field_name_to_type: dict[str, Any]) -> type:
    return _create_dataclass(new_type_name, tuple(bases), tuple(sorted(field_name_to_type.items())))


@functools.cache
def _create_dataclass(new_type_name: str, bases: tuple[type], field_name_to_type: tuple[tuple[str, Any], ...]) -> type:
    new_type: type = create_type(new_type_name, bases, {name: None for name in field_name_to_type})
    new_type.__annotations__ = dict(field_name_to_type)
    # noinspection PyTypeChecker
    new_dataclass = dataclasses.dataclass(new_type)
    return new_dataclass


@dataclass(unsafe_hash=True)
class View:
    optional: InitVar[Iterable[Union[sqlalchemy.Column, sqlalchemy.orm.attributes.InstrumentedAttribute]]] = None
    required: InitVar[Iterable[Union[sqlalchemy.Column, sqlalchemy.orm.attributes.InstrumentedAttribute]]] = None
    excluded: InitVar[Iterable[Union[sqlalchemy.Column, sqlalchemy.orm.attributes.InstrumentedAttribute]]] = None
    excluded_relations: frozenset[sqlalchemy.orm.RelationshipProperty] = dataclasses.field(init=False)
    excluded_columns: frozenset[sqlalchemy.Column] = dataclasses.field(init=False)
    required_columns: frozenset[sqlalchemy.Column] = dataclasses.field(init=False)
    optional_columns: frozenset[sqlalchemy.Column] = dataclasses.field(init=False)

    @staticmethod
    def is_instance(possible_instance: 'ViewableEntity') -> bool:
        return False

    def __repr__(self):
        cls = type(self)
        values = []
        if self.excluded_columns or self.excluded_relations:
            excluded_strings = [str(rel.key) for rel in itertools.chain(self.excluded_columns, self.excluded_relations)]
            values.append(f'excluded={",".join(excluded_strings)}')
        if self.required_columns:
            values.append(f'required={",".join([str(rel.key) for rel in self.required_columns])}')
        if self.optional_columns:
            values.append(f'optional={",".join([str(rel.key) for rel in self.optional_columns])}')
        return f'{cls.__name__}({",".join(values)})'

    @classmethod
    def process_field_name_to_field(cls, entity_type: Type['ViewableEntity'], field_name_to_field: dict[str, dataclasses.field]
                                    ) -> dict[str, dataclasses.field]:
        return cls._process_field_name_to_field(entity_type, field_name_to_field)

    @classmethod
    @abc.abstractmethod
    def _process_field_name_to_field(
            cls, entity_type: Type['ViewableEntity'], field_name_to_field: dict[str, dataclasses.field],
            required_columns: frozenset[sqlalchemy.Column] = None,
            optional_columns: frozenset[sqlalchemy.Column] = None,
            excluded_columns: frozenset[sqlalchemy.Column] = None,
            excluded_relations: frozenset[sqlalchemy.orm.RelationshipProperty] = None) -> dict[str, dataclasses.field]:
        pass

    def _process_field_name_to_field__instance(self, entity_type: Type['ViewableEntity'],
                                               field_name_to_field: dict[str, dataclasses.field]
                                               ) -> dict[str, dataclasses.field]:
        return self._process_field_name_to_field(
            entity_type, field_name_to_field, excluded_relations=self.excluded_relations,
            required_columns=self.required_columns, optional_columns=self.optional_columns,
            excluded_columns=self.excluded_columns
        )

    def __post_init__(self,
                      optional: Iterable[Union[sqlalchemy.Column, sqlalchemy.orm.attributes.InstrumentedAttribute]],
                      required: Iterable[Union[sqlalchemy.Column, sqlalchemy.orm.attributes.InstrumentedAttribute]],
                      excluded: Iterable[sqlalchemy.orm.attributes.InstrumentedAttribute]):
        optional_columns = []
        if optional:
            for attribute in optional:
                if (column := as_type(attribute, sqlalchemy.Column)) is not None:
                    optional_columns.append(column)
                    continue

                property_ = attribute.prop
                if (relation := as_type(property_, sqlalchemy.orm.RelationshipProperty)) is not None:
                    raise TypeError(f"Relations are not supported by optional/required: {relation}")
                elif (column_property := as_type(property_, sqlalchemy.orm.ColumnProperty)) is not None:
                    column = column_property.expression
                    optional_columns.append(column)
                else:
                    raise TypeError("Unexpected type")
        self.optional_columns = frozenset(optional_columns)

        required_columns = []
        if required:
            for attribute in required:
                if (column := as_type(attribute, sqlalchemy.Column)) is not None:
                    required_columns.append(column)
                    continue

                property_ = attribute.prop
                if (relation := as_type(property_, sqlalchemy.orm.RelationshipProperty)) is not None:
                    raise TypeError(f"Relations are not supported by optional/required, got: {relation}")
                elif (column_property := as_type(property_, sqlalchemy.orm.ColumnProperty)) is not None:
                    column = column_property.expression
                    required_columns.append(column)
                else:
                    raise TypeError("Unexpected type")
        self.required_columns = frozenset(required_columns)

        excluded_relations = []
        excluded_columns = []
        excluded = to_list(excluded)
        if excluded:
            for attribute in excluded:
                if (column := as_type(attribute, sqlalchemy.Column)) is not None:
                    excluded_columns.append(column)
                    continue

                property_ = attribute.prop
                if (relation := as_type(property_, sqlalchemy.orm.RelationshipProperty)) is not None:
                    excluded_relations.append(relation)
                elif (column_property := as_type(property_, sqlalchemy.orm.ColumnProperty)) is not None:
                    column = column_property.expression
                    excluded_columns.append(column)
                    # raise TypeError(f"Columns are not supported by exclude, got: {column}")
                else:
                    raise TypeError("Unexpected type")
        self.excluded_relations = frozenset(excluded_relations)
        self.excluded_columns = frozenset(excluded_columns)

        self.process_field_name_to_field = self._process_field_name_to_field__instance


# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #


T = TypeVar("T")


class RestrictionViolation(Exception):
    pass


@dataclass
class EntityDataRestriction:
    condition: Callable[['ViewableEntity'], Any]
    name: Optional[str] = None
    exception: Exception = RestrictionViolation
    scope_limited_by_view_types: Optional[list[Type[View]]] = None

    def check(self, entity: 'ViewableEntity') -> None:
        if self.scope_limited_by_view_types is not None:
            if not any(view_type.is_instance(entity) for view_type in self.scope_limited_by_view_types):
                return

        if error_description := self.condition(entity):
            raise self.exception(error_description)


def _short_repr(self: 'ViewableEntity') -> str:
    attrs = [
        f'{name}={value!r}' for name in self.get_column_names()
        if (value := getattr(self, name)) not in (NoValue, NoDefault)
    ]
    if self.is_delta_view():
        attrs += [
            f'len({name}:c/u/d)={len(value.create)}/{len(value.update)}/{len(value.delete)}'
            for name in self.get_relation_names()
            if (value := getattr(self, name)) not in (NoValue, NoDefault)
        ]
    else:
        attrs += [
            f'len({name})={len(value)}' for name in self.get_relation_names()
            if (value := getattr(self, name)) not in (NoValue, NoDefault)
        ]

    return f'{type(self).__name__}({", ".join(attrs)})'


class ViewableEntity(ReprInDumps):
    _parent_view: Union['ViewableEntity', NoValue] = NoValue
    _cls_to_restrictions: dict[Type['ViewableEntity'], list[EntityDataRestriction]] = {}

    @classmethod
    def add_restriction(cls, restriction: EntityDataRestriction) -> None:
        cls._cls_to_restrictions.setdefault(cls.get_entity_type(), []).append(restriction)

    def _check_restrictions(self) -> None:
        for restriction in self._cls_to_restrictions.get(self.get_entity_type(), []):
            restriction.check(self)

    def __post_init__(self):
        self._check_restrictions()

    def get_parent_view(self):
        return self._parent_view

    @classmethod
    def get_entity_type(cls) -> Type['ViewableEntity']:
        return cls

    def __repr_in_dumps__(self) -> dict[str, Any]:
        return self.as_dict(plain=False)

    @classmethod
    def is_insert_view(cls) -> bool:
        return False

    @classmethod
    def is_update_view(cls) -> bool:
        return False

    @classmethod
    def is_selected_view(cls) -> bool:
        return False

    @classmethod
    def is_delete_view(cls) -> bool:
        return False

    @classmethod
    def is_delta_view(cls) -> bool:
        return False

    @classmethod
    def is_full_view(cls) -> bool:
        return False

    @classmethod
    def is_not_empty(cls) -> bool:
        return False

    @classmethod
    @functools.cache
    def get_table(cls) -> sqlalchemy.Table:
        # noinspection PyUnresolvedReferences
        return cls.__table__

    @classmethod
    @functools.cache
    def get_table_name(cls) -> str:
        return cls.get_table().name

    @classmethod
    @functools.cache
    def get_key_to_relation(cls) -> Mapping[str, sqlalchemy.orm.RelationshipProperty]:
        # noinspection PyUnresolvedReferences
        return MappingProxyType(cls.__mapper__.relationships)

    @classmethod
    @functools.cache
    def get_relation_names(cls) -> tuple[str, ...]:
        return tuple(cls.get_key_to_relation().keys())

    @classmethod
    @functools.cache
    def get_primary_key_columns(cls) -> tuple[sqlalchemy.Column, ...]:
        # noinspection PyUnresolvedReferences
        return tuple(cls.__mapper__.mapper.primary_key)

    def get_primary_key_values(self) -> tuple[Any, ...]:
        return tuple(getattr(self, name) for name in self.get_primary_key_names())

    def get_primary_key_name_to_value(self) -> dict[str, Any]:
        return {name: getattr(self, name) for name in self.get_primary_key_names()}

    # def set_primary_key_values(self, values: tuple[Any]) -> None:
    #     for name, val in zip(self.get_primary_key_names(), values):
    #         setattr(self, name, val)

    @classmethod
    @functools.cache
    def get_primary_key_names(cls) -> tuple[str, ...]:
        return tuple(col.name for col in cls.get_primary_key_columns())

    @classmethod
    @functools.cache
    def get_columns(cls) -> tuple[Column, ...]:
        # noinspection PyUnresolvedReferences
        return tuple(cls.__mapper__.mapper.columns)

    @classmethod
    @functools.cache
    def get_identifier_columns(cls) -> tuple[Column, ...]:
        fields = dataclasses.fields(cls)
        values = []
        for field in fields:
            if field.metadata.get("identifier") and (field.repr or field.init):
                values.append(field.metadata['sa'])
        return tuple(values)
        return tuple(
            getattr(cls, field.name)
            for field in dataclasses.fields(cls)
            if field.metadata.get("identifier") and (field.repr or field.init)
        )

    def get_identifier_value_dict(self) -> dict[str, Any]:
        return {column.name: getattr(self, column.name) for column in self.get_identifier_columns()}

    def get_identifier_value_tuple(self) -> tuple[Any]:
        columns = self.get_identifier_columns()
        values = []
        for column in columns:
            values.append(getattr(self, column.name))
        return tuple(values)
        return tuple(getattr(self, column.name) for column in self.get_identifier_columns())

    @classmethod
    @functools.cache
    def get_non_primary_key_columns(cls) -> tuple[Column, ...]:
        # noinspection PyUnresolvedReferences,PyTypeChecker
        return tuple(col for col in cls.__mapper__.mapper.columns if col.name not in cls.get_primary_key_names())

    @classmethod
    @functools.cache
    def get_column_names(cls) -> tuple[str, ...]:
        # noinspection PyTypeChecker
        return tuple(col.name for col in cls.get_columns())

    def as_dict(self, plain: bool = False) -> dict[str, Any]:
        result = {}
        for column in self.get_columns():
            value = getattr(self, column.name)
            if value in (NoValue, NoDefault) or value is None and not column.nullable:
                continue
            result[column.name] = value

        if plain:
            return result

        for relation_name, relation in self.get_key_to_relation().items():
            value = getattr(self, relation_name)
            if value in (NoValue, NoDefault):
                continue
            if self.is_delta_view():
                result[relation_name] = {
                    "create": [val.as_dict() for val in value.create] if relation.uselist else value.create.as_dict(),
                    "update": [val.as_dict() for val in value.update] if relation.uselist else value.update.as_dict(),
                    "delete": [val.as_dict() for val in value.delete] if relation.uselist else value.delete.as_dict(),
                }
            else:
                result[relation_name] = [val.as_dict() for val in value] if relation.uselist else value.as_dict()
        return result

    @classmethod
    @functools.cache
    def produce_view(cls: Type[T], view: Union[View, Type[View]]) -> Type[T]:
        start_time = time.time()
        view_type = view if isinstance(view, type) else type(view)
        logger.debug(f'produce_view(%s, %s)', cls.__name__, view_type.__name__)
        if not dataclasses.is_dataclass(cls):
            raise TypeError('must be called with a dataclass type or instance')

        field_name_to_field = copy.copy(get_dataclass_field_name_to_field(cls))
        field_name_to_field = view.process_field_name_to_field(cls, field_name_to_field)

        field_name_to_type = {
            name: field.type
            for name, field in field_name_to_field.items()
            if isinstance(field, dataclasses.Field)
        }
        field_name_to_field['get_entity_type'] = lambda x=None: cls
        field_name_to_field["_parent_view"] = dataclasses.field(repr=False, default=NoValue)
        field_name_to_field["__str__"] = field_name_to_field["__repr__"] = _short_repr
        field_name_to_type["_parent_view"] = Optional[cls]
        # field_name_to_field = {}
        # TODO: use create_dataclass here, cons: requires signature expand
        new_type: type = create_type(
            _prepare_class_name([cls], view), [cls],
            field_name_to_field)
        new_type.__annotations__ = field_name_to_type
        # noinspection PyTypeChecker
        new_dataclass = dataclasses.dataclass(new_type)
        logger.debug(f'produced %s in %s ms', new_dataclass.__name__, round((time.time() - start_time) * 1000, 2))
        return new_dataclass

    def get_types_and_objects_amount(self) -> tuple[int, int]:
        type_to_primary_keys_to_entity = self.get_type_to_primary_keys_to_entity(use_parent=False)
        return (
            len(type_to_primary_keys_to_entity),
            sum(len(values) for values in type_to_primary_keys_to_entity.values())
        )

    # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
    # insert_default ->

    @classmethod
    @functools.cache
    def get_key_to_insert_default(cls) -> Mapping[str, Callable[[...], Awaitable[Any]]]:
        # noinspection PyDataclass
        return MappingProxyType({
            field.name: insert_default
            for field in dataclasses.fields(cls)
            if (insert_default := field.metadata.get("insert_default"))
        })

    @classmethod
    @functools.cache
    def get_key_to_relation_with_insert_default(cls) -> Mapping[str, Callable[[...], Awaitable[Any]]]:
        return MappingProxyType({
            key: relation
            for key, relation in cls.get_key_to_relation().items()
            if get_related_entity_type(relation).has_insert_defaults()
        })

    @classmethod
    @functools.cache
    def has_insert_defaults(cls) -> bool:
        return bool(cls.get_key_to_insert_default() or cls.get_key_to_relation_with_insert_default())

    async def resolve_insert_defaults(self, context) -> None:
        for key, insert_default in self.get_key_to_insert_default().items():
            logger.debug("async default: %s, %s", type(self), key)
            setattr(self, key, await acall(insert_default(context)))

        for key, relation in self.get_key_to_relation_with_insert_default().items():
            value = getattr(self, key)
            if relation.uselist:
                if isinstance(value, list):
                    logger.debug("insert default one-to_many relation: %s, %s", type(self), key)
                    await asyncio.gather(*[val.resolve_insert_defaults(context) for val in value])
            elif isinstance(value, ViewableEntity):
                logger.debug("insert default ono-to-one relation: %s, %s", type(self), key)
                await value.resolve_insert_defaults(context)
        return

    # <- insert_default
    # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #

    def _save_type_to_primary_keys_to_entity__to(self, result: dict[Type['ViewableEntity'], dict[tuple, 'ViewableEntity']]) -> None:
        result.setdefault(self.get_entity_type(), {})[self.get_primary_key_values()] = self

        for relation_name, relation in self.get_key_to_relation().items():
            value = getattr(self, relation_name)
            if value in (NoValue, NoDefault):
                continue
            if self.is_delta_view():
                related_entities = itertools.chain(value.create, value.update, value.delete)
            else:
                related_entities = value if relation.uselist else [value]

            for entity in related_entities:
                entity._save_type_to_primary_keys_to_entity__to(result)

    def get_type_to_primary_keys_to_entity(self,
                                           use_parent: bool = True) -> dict[Type['ViewableEntity'], dict[tuple, 'ViewableEntity']]:
        result = {}
        self._save_type_to_primary_keys_to_entity__to(result)
        if use_parent and self._parent_view is not NoValue:
            self._parent_view._save_type_to_primary_keys_to_entity__to(result)

        return result

    def clean_primary_keys_with_default(self) -> None:
        for primary_key_column in self.get_primary_key_columns():
            if has_default(primary_key_column):
                setattr(self, primary_key_column.name, NoValue)
        for relation_name in self.get_relation_names():
            related_entities = getattr(self, relation_name)
            if related_entities not in (NoValue, NoDefault):
                for entity in related_entities:
                    entity.clean_primary_keys_with_default()

    # @functools.cache
    def __class_getitem__(cls: Type[T], view_type: Union[View, Type[View]]) -> Type[T]:
        return cls.produce_view(view_type)

# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #


def get_column_pairs_from_relation(relation: sqlalchemy.orm.RelationshipProperty
                                   ) -> list[tuple[sqlalchemy.Column, sqlalchemy.Column]]:
    return relation.synchronize_pairs


def is_auto_incremented(column: sqlalchemy.Column) -> bool:
    """
    some logic ignored, may be done later
    https://docs.sqlalchemy.org/en/14/core/metadata.html#sqlalchemy.schema.Column.params.autoincrement
    """
    if column.primary_key and isinstance(column.type, Integer) and not column.foreign_keys:
        return bool(column.autoincrement)


def has_client_default(column: sqlalchemy.Column) -> bool:
    return column.default is not None


def has_server_default(column: sqlalchemy.Column) -> bool:
    return column.server_default is not None or is_auto_incremented(column)


def has_default(column: sqlalchemy.Column):
    return has_client_default(column) or has_server_default(column)


def get_sql_alchemy_property(field: dataclasses.field):
    return field.metadata['sa']


def as_relation(attribute) -> Optional[sqlalchemy.orm.RelationshipProperty]:
    if isinstance(attribute, sqlalchemy.orm.RelationshipProperty):
        return attribute


def as_column(attribute) -> Optional[sqlalchemy.Column]:
    if isinstance(attribute, sqlalchemy.Column):
        return attribute


def as_type(val: Any, type_: Type[T]) -> Optional[T]:
    if isinstance(val, type_):
        return val


def get_related_entity_type(relation: sqlalchemy.orm.RelationshipProperty) -> Type[ViewableEntity]:
    return relation.argument


def is_list_relation(relation: sqlalchemy.orm.RelationshipProperty) -> bool:
    return relation.uselist


def get_remote_column(relation: sqlalchemy.orm.RelationshipProperty) -> sqlalchemy.Column:
    class_attribute: sqlalchemy.orm.attributes.InstrumentedAttribute = relation.class_attribute
    expression = as_type(class_attribute.expression, sqlalchemy.sql.elements.BinaryExpression)
    if expression is None:
        raise TypeError(f"failed to process relation: {relation}")

    remote_column: sqlalchemy.Column = expression.right  # could it be left?
    return remote_column


def get_local_and_remote_column(relation: sqlalchemy.orm.RelationshipProperty
                                ) -> Tuple[sqlalchemy.Column, sqlalchemy.Column]:
    class_attribute: sqlalchemy.orm.attributes.InstrumentedAttribute = relation.class_attribute
    expression = as_type(class_attribute.expression, sqlalchemy.sql.elements.BinaryExpression)
    if expression is None:
        raise TypeError(f"failed to process relation: {relation}")

    local_column: sqlalchemy.Column = expression.left  # could it be right?
    remote_column: sqlalchemy.Column = expression.right  # could it be left?
    return local_column, remote_column

# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #


@dataclass(unsafe_hash=True, repr=False)
class InsertView(View):
    @staticmethod
    def is_instance(possible_instance: 'ViewableEntity') -> bool:
        if possible_instance.is_insert_view():
            return True

    @classmethod
    @abc.abstractmethod
    def _process_field_name_to_field(cls, entity_type: Type[ViewableEntity], field_name_to_field: dict[str, dataclasses.field],
                                     required_columns: frozenset[sqlalchemy.Column] = None,
                                     optional_columns: frozenset[sqlalchemy.Column] = None,
                                     excluded_columns: frozenset[sqlalchemy.Column] = None,
                                     excluded_relations: frozenset[sqlalchemy.orm.RelationshipProperty] = None,
                                     ) -> dict[str, dataclasses.field]:
        """
        all not_nullable columns should get value, either from incoming data or from defaults
        all client side defaults are already passed to model, but there are DB server_defaults and numeric primary keys
        incremental primary keys are optional

        https://docs.sqlalchemy.org/en/14/core/metadata.html#sqlalchemy.schema.Column.params.autoincrement
        integer primary key column with no foreign key dependencies
        """
        required_columns = required_columns if required_columns else []
        optional_columns = optional_columns if optional_columns else []
        excluded_columns = excluded_columns if excluded_columns else []
        excluded_relations = excluded_relations if excluded_relations else []

        for field_name, field in field_name_to_field.items():
            field_copy = copy.copy(field)
            sql_alchemy_property = get_sql_alchemy_property(field)

            if (relation := as_type(sql_alchemy_property, sqlalchemy.orm.RelationshipProperty)) is not None:
                field_copy.default = NoValue
                field_copy.default_factory = dataclasses.MISSING
                if relation in excluded_relations:
                    field_copy.repr = False
                    field_copy.init = False
                else:
                    RelatedEntity = get_related_entity_type(relation)
                    remote_column = get_remote_column(relation)
                    related_entity_view_type = RelatedEntity[InsertView(excluded=[remote_column])]
                    is_list = is_list_relation(relation)
                    field_copy.type = list[related_entity_view_type] if is_list else related_entity_view_type
            elif (column := as_type(sql_alchemy_property, sqlalchemy.Column)) is not None:
                # if field
                if column in required_columns:
                    field_copy.default = NoDefault
                elif column in optional_columns:
                    field_copy.default = NoValue
                elif column in excluded_columns:
                    field_copy.default = NoValue
                    field_copy.repr = False
                    field_copy.init = False
                elif field.default in (dataclasses.MISSING, NoValue) and has_default(column):
                    field_copy.default = NoValue
                    if not column.nullable:
                        field_copy.type = Optional[field_copy.type]
            else:
                logger.warning(f"ignored attribute: %s, cos %s has unexpected type", field, sql_alchemy_property)

            field_name_to_field[field_name] = field_copy

        # for field_name in list(field_name_to_field):
        #     if field_name_to_field[field_name] is None:
        #         del field_name_to_field[field_name]
        field_name_to_field['is_insert_view'] = lambda x: True
        return field_name_to_field


@dataclass(unsafe_hash=True, repr=False)
class UpdateView(View):
    @staticmethod
    def is_instance(possible_instance: 'ViewableEntity') -> bool:
        return possible_instance.is_update_view()

    @classmethod
    def _process_field_name_to_field(cls, entity_type: Type[ViewableEntity], field_name_to_field: dict[str, dataclasses.field],
                                     required_columns: frozenset[sqlalchemy.Column] = None,
                                     optional_columns: frozenset[sqlalchemy.Column] = None,
                                     excluded_columns: frozenset[sqlalchemy.Column] = None,
                                     excluded_relations: frozenset[sqlalchemy.orm.RelationshipProperty] = None,
                                     ) -> dict[str, dataclasses.field]:
        """Non-primary keys are optional, all primary_keys - required
        """
        required_columns = required_columns if required_columns else []
        optional_columns = optional_columns if optional_columns else []
        excluded_columns = excluded_columns if excluded_columns else []
        excluded_relations = excluded_relations if excluded_relations else []

        for field_name, field in field_name_to_field.items():
            field_copy = copy.copy(field)
            sql_alchemy_property = get_sql_alchemy_property(field)

            if (relation := as_relation(sql_alchemy_property)) is not None:
                field_copy.default = NoValue
                field_copy.default_factory = dataclasses.MISSING
                if relation in excluded_relations:
                    field_copy.repr = False
                    field_copy.init = False
                else:
                    RelatedEntity = get_related_entity_type(relation)
                    remote_column = get_remote_column(relation)
                    related_entity_view_type = RelatedEntity[InsertView(excluded=[remote_column])]
                    is_list = is_list_relation(relation)
                    field_copy.type = list[related_entity_view_type] if is_list else related_entity_view_type
            elif (column := as_type(sql_alchemy_property, sqlalchemy.Column)) is not None:
                if column in required_columns:
                    field_copy.default = NoDefault
                elif column in optional_columns:
                    field_copy.default = NoValue
                elif column in excluded_columns:
                    field_copy.default = NoValue
                    field_copy.repr = False
                    field_copy.init = False
                else:
                    field_copy.default = NoDefault if sql_alchemy_property.primary_key else NoValue
            else:
                logger.warning(f"ignored attribute: %s, cos %s has unexpected type", field, sql_alchemy_property)

            field_name_to_field[field_name] = field_copy

        field_name_to_field['is_update_view'] = lambda x: True
        return field_name_to_field


class SelectedView(View):
    @staticmethod
    def is_instance(possible_instance: 'ViewableEntity') -> bool:
        return possible_instance.is_selected_view()

    @classmethod
    def _process_field_name_to_field(cls, entity_type: Type[ViewableEntity], field_name_to_field: dict[str, dataclasses.field],
                                     required_columns: frozenset[sqlalchemy.Column] = None,
                                     optional_columns: frozenset[sqlalchemy.Column] = None,
                                     excluded_columns: frozenset[sqlalchemy.Column] = None,
                                     excluded_relations: frozenset[sqlalchemy.orm.RelationshipProperty] = None,
                                     ) -> dict[str, dataclasses.field]:
        """Everything is optional
        """
        required_columns = required_columns if required_columns else []
        excluded_columns = excluded_columns if excluded_columns else []
        excluded_relations = excluded_relations if excluded_relations else []

        for field_name, field in field_name_to_field.items():
            field_copy = copy.copy(field)
            sql_alchemy_property = get_sql_alchemy_property(field)

            if (relation := as_relation(sql_alchemy_property)) is not None:
                field_copy.default = NoValue
                field_copy.default_factory = dataclasses.MISSING
                if relation in excluded_relations:
                    field_copy.repr = False
                    field_copy.init = False
                else:
                    RelatedEntity = get_related_entity_type(relation)
                    related_entity_view_type = RelatedEntity[SelectedView]
                    is_list = is_list_relation(relation)
                    field_copy.type = list[related_entity_view_type] if is_list else related_entity_view_type
            elif (column := as_type(sql_alchemy_property, sqlalchemy.Column)) is not None:
                if column in excluded_columns:
                    field_copy.default = NoValue
                    field_copy.repr = False
                    field_copy.init = False
                else:
                    field_copy.default = NoDefault if column in required_columns else NoValue
            else:
                logger.warning(f"ignored attribute: %s, cos %s has unexpected type", field, sql_alchemy_property)

            # if field_copy.default == NoValue:
            #     field_copy.type = Optional[field_copy.type]
            field_name_to_field[field_name] = field_copy

        field_name_to_field['is_selected_view'] = lambda x: True

        return field_name_to_field


@dataclass(unsafe_hash=True, repr=False)
class DeleteView(View):
    @staticmethod
    def is_instance(possible_instance: 'ViewableEntity') -> bool:
        return possible_instance.is_delete_view()

    @classmethod
    def _process_field_name_to_field(cls, entity_type: Type[ViewableEntity], field_name_to_field: dict[str, dataclasses.field],
                                     required_columns: frozenset[sqlalchemy.Column] = None,
                                     optional_columns: frozenset[sqlalchemy.Column] = None,
                                     excluded_columns: frozenset[sqlalchemy.Column] = None,
                                     excluded_relations: frozenset[sqlalchemy.orm.RelationshipProperty] = None,
                                     ) -> dict[str, dataclasses.field]:
        """Non-primary keys are optional, all primary_keys - required
        """
        excluded_columns = excluded_columns if excluded_columns else []
        if required_columns or optional_columns or excluded_relations:
            raise TypeError("DeletedView doesn't support required_columns, optional_columns or excluded_relations")

        for field_name, field in field_name_to_field.items():
            field_copy = copy.copy(field)
            sql_alchemy_property = get_sql_alchemy_property(field)

            if as_relation(sql_alchemy_property) is not None:
                field_copy.default = NoValue
                field_copy.default_factory = dataclasses.MISSING
                field_copy.repr = False
                field_copy.init = False
            elif as_type(sql_alchemy_property, sqlalchemy.Column) is not None:
                if sql_alchemy_property.primary_key and field_name not in excluded_columns:
                    field_copy.default = NoDefault
                else:
                    field_copy.default = NoValue
                    field_copy.repr = False
                    field_copy.init = False
            else:
                logger.warning(f"ignored attribute: %s, cos %s has unexpected type", field, sql_alchemy_property)

            field_name_to_field[field_name] = field_copy

        field_name_to_field['is_delete_view'] = lambda x: True
        return field_name_to_field


class DeltaView(View):
    @staticmethod
    def is_instance(possible_instance: 'ViewableEntity') -> bool:
        return possible_instance.is_delta_view()

    @classmethod
    def _process_field_name_to_field(cls, entity_type: Type[ViewableEntity], field_name_to_field: dict[str, dataclasses.field],
                                     required_columns: frozenset[sqlalchemy.Column] = None,
                                     optional_columns: frozenset[sqlalchemy.Column] = None,
                                     excluded_columns: frozenset[sqlalchemy.Column] = None,
                                     excluded_relations: frozenset[sqlalchemy.orm.RelationshipProperty] = None,
                                     ) -> dict[str, dataclasses.field]:
        """Everything is optional
        """
        excluded_columns = excluded_columns if excluded_columns else []
        if required_columns or optional_columns or excluded_relations:
            raise TypeError("DeletedView doesn't support required_columns, optional_columns or excluded_relations")

        for field_name, field in field_name_to_field.items():
            field_copy = copy.copy(field)
            sql_alchemy_property = get_sql_alchemy_property(field)

            if relation := as_relation(sql_alchemy_property):
                field_copy.default = NoValue
                field_copy.default_factory = dataclasses.MISSING
                RelatedEntity = get_related_entity_type(relation)
                remote_column = get_remote_column(relation)
                if is_list_relation(relation):
                    field_copy.type = create_dataclass(
                        _prepare_class_name([RelatedEntity], 'delta'), [], {
                            "create": list[RelatedEntity[InsertView(excluded=[remote_column])]],
                            "update": list[RelatedEntity[DeltaView(excluded=[remote_column])]],
                            "delete": list[RelatedEntity[DeleteView(excluded=[remote_column])]],
                        }
                    )
                else:
                    field_copy.type = Union[
                        RelatedEntity[InsertView(excluded=[remote_column])],
                        RelatedEntity[DeltaView(excluded=[remote_column])],
                        RelatedEntity[DeleteView(excluded=[remote_column])]
                    ]
            elif (column := as_type(sql_alchemy_property, sqlalchemy.Column)) is not None:
                if field_name in excluded_columns:
                    field_copy.default = NoValue
                    field_copy.repr = False
                    field_copy.init = False
                elif column.primary_key:
                    field_copy.default = NoDefault
                else:
                    field_copy.type = create_dataclass(
                        _prepare_class_name([field_copy.type], 'delta'), [], {
                            "old": field_copy.type,
                            "new": field_copy.type
                        }
                    )
                    field_copy.default = NoValue
            else:
                logger.warning(f"ignored attribute: %s, cos %s has unexpected type", field, sql_alchemy_property)

            # if field_copy.default == NoValue:
            #     field_copy.type = Optional[field_copy.type]
            field_name_to_field[field_name] = field_copy

        field_name_to_field['is_delta_view'] = lambda x: True
        field_name_to_field['is_not_empty'] = cls.is_not_empty
        return field_name_to_field

    @staticmethod
    def is_not_empty(entity: 'ViewableEntity[DeltaView]'):
        if any(
                value is not None or not column.nullable
                for column in entity.get_non_primary_key_columns()
                if (value := getattr(entity, column.name)) not in (NoValue, NoDefault)
        ):
            return True

        for relation_name, relation in entity.get_key_to_relation().items():
            value = getattr(entity, relation_name)
            if value in (NoValue, NoDefault):
                continue
            if (create := value.create) and create not in (NoValue, NoDefault):
                return True
            if (delete := value.delete) and delete not in (NoValue, NoDefault):
                return True
            if (update := value.update) and update not in (NoValue, NoDefault):
                if any(val.is_not_empty() for val in value.update) if relation.uselist else value.update.is_not_empty():
                    return True

        return False


class FullView(View):
    @staticmethod
    def is_instance(possible_instance: 'ViewableEntity') -> bool:
        return possible_instance.is_full_view()

    @classmethod
    def _process_field_name_to_field(cls, entity_type: Type[ViewableEntity], field_name_to_field: dict[str, dataclasses.field],
                                     required_columns: frozenset[sqlalchemy.Column] = None,
                                     optional_columns: frozenset[sqlalchemy.Column] = None,
                                     excluded_columns: frozenset[sqlalchemy.Column] = None,
                                     excluded_relations: frozenset[sqlalchemy.orm.RelationshipProperty] = None,
                                     ) -> dict[str, dataclasses.field]:
        """Everything is required
        """
        excluded_columns = excluded_columns if excluded_columns else []
        # TODO: allow to exclude only foreign key columns
        if required_columns or optional_columns or excluded_relations:
            raise TypeError("FullView doesn't support required_columns, optional_columns or excluded_relations")

        for field_name, field in field_name_to_field.items():
            field_copy = copy.copy(field)
            sql_alchemy_property = get_sql_alchemy_property(field)

            if relation := as_relation(sql_alchemy_property):
                field_copy.default = NoValue
                field_copy.default_factory = dataclasses.MISSING

                RelatedEntity = get_related_entity_type(relation)
                related_entity_view_type = RelatedEntity[FullView]
                is_list = is_list_relation(relation)
                field_copy.type = list[related_entity_view_type] if is_list else related_entity_view_type
            elif as_type(sql_alchemy_property, sqlalchemy.Column) is not None:
                if field_name in excluded_columns:
                    field_copy.default = NoValue
                    field_copy.repr = False
                    field_copy.init = False
                else:
                    field_copy.default = NoDefault
            else:
                logger.warning(f"ignored attribute: %s, cos %s has unexpected type", field, sql_alchemy_property)

            field_name_to_field[field_name] = field_copy

        field_name_to_field['is_full_view'] = lambda x: True
        return field_name_to_field

    @staticmethod
    def get_delta(self: 'ViewableEntity[FullView]', other: 'ViewableEntity[UpdateView]') -> 'ViewableEntity[DeltaView]':
        # TODO: maybe move this from FullView, but delta could be rendered for sure only based on FullView
        delta_key_to_value = {"_parent_view": self}
        primary_key_names = self.get_primary_key_names()
        for column_name in primary_key_names:
            self_value = getattr(self, column_name)
            other_value = getattr(other, column_name)
            if other_value not in (NoValue, NoDefault):
                assert self_value == other_value
            delta_key_to_value[column_name] = self_value

        for column_name in set(self.get_column_names()) - set(primary_key_names):
            other_value = getattr(other, column_name)
            self_value = getattr(self, column_name)
            if other_value not in (NoValue, NoDefault) and other_value != self_value:
                delta_key_to_value[column_name] = {"old": self_value, "new": other_value}

        for relation_name, relation in self.get_key_to_relation().items():
            other_value = getattr(other, relation_name)
            if other_value is not NoValue and other_value is not NoDefault:
                self_value = getattr(self, relation_name)
                if not relation.uselist:
                    other_value = [other_value]
                    self_value = [self_value]

                for val in self_value:
                    val._parent_view = val  # to make possible get values from delete view

                related_entity_type = get_related_entity_type(relation)
                remote_column = get_remote_column(relation)
                non_relational_primary_key_names = [
                    col.key for col in related_entity_type.get_primary_key_columns() if col != remote_column]

                self_relation_primary_keys_to_value = {
                    tuple(getattr(val, name) for name in non_relational_primary_key_names): val for val in self_value}

                other_values_without_primary_keys = []
                other_relation_primary_keys_to_value = {}
                for val in other_value:
                    primary_key_values = tuple(getattr(val, name) for name in non_relational_primary_key_names)
                    if None in primary_key_values or NoValue in primary_key_values or NoDefault in primary_key_values:
                        other_values_without_primary_keys.append(val)
                    else:
                        other_relation_primary_keys_to_value[primary_key_values] = val

                required_but_missing_relation_values = [
                    val
                    for keys, val in other_relation_primary_keys_to_value.items()
                    if keys not in self_relation_primary_keys_to_value
                ] + other_values_without_primary_keys
                existent_but_unused_relation_values = [
                    # dict_to_dataclass(val, related_entity_type[DeleteView])
                    val
                    for keys, val in self_relation_primary_keys_to_value.items()
                    if keys not in other_relation_primary_keys_to_value
                ]
                updated_relation_values = [
                    delta
                    for keys, val in self_relation_primary_keys_to_value.items()
                    if keys in other_relation_primary_keys_to_value
                    and (delta := FullView.get_delta(val, other_relation_primary_keys_to_value[keys])).is_not_empty()
                ]

                # compare values by alternative identifier
                identifier_column_names = [
                    col.name for col in related_entity_type.get_identifier_columns() if col != remote_column
                ]
                if identifier_column_names:
                    identifier_to_required_but_missing_relation_values = {}
                    for required in other_values_without_primary_keys:
                        identifier: tuple = tuple(getattr(required, name) for name in identifier_column_names)
                        identifier_to_required_but_missing_relation_values.setdefault(identifier, []).append(required)

                    for existent in existent_but_unused_relation_values[:]:  # copy, coz is changed in iteration
                        identifier: tuple = tuple(getattr(existent, name) for name in identifier_column_names)
                        candidates = identifier_to_required_but_missing_relation_values.get(identifier, [])
                        if not candidates:
                            continue
                        best_candidate = None
                        best_candidate_delta = None
                        for candidate in candidates:
                            candidate_delta = FullView.get_delta(existent, candidate)
                            if best_candidate_delta is None:
                                best_candidate_delta, best_candidate = candidate_delta, candidate
                            else:
                                candidate_delta_complexity = candidate_delta.get_types_and_objects_amount()
                                # TODO: use cache here somehow
                                best_candidate_delta_complexity = best_candidate_delta.get_types_and_objects_amount()
                                if candidate_delta_complexity < best_candidate_delta_complexity:
                                    best_candidate_delta, best_candidate = candidate_delta, candidate

                        # logger.trace(f"found value by identifier:")
                        # logger.trace(f"existent: {existent}")
                        # logger.trace(f"suitable for {best_candidate}")
                        # logger.trace(f"with delta {best_candidate_delta}")
                        # logger.trace("---")
                        candidates.remove(best_candidate)
                        if best_candidate_delta.is_not_empty():
                            updated_relation_values.append(best_candidate_delta)
                        required_but_missing_relation_values.remove(best_candidate)
                        existent_but_unused_relation_values.remove(existent)

                delta_key_to_value[relation_name] = {
                    "create": required_but_missing_relation_values,
                    "update": updated_relation_values,
                    "delete": existent_but_unused_relation_values
                }

        left_entity_type = self.get_entity_type()
        delta_view_type = left_entity_type[DeltaView]
        delta_view = dict_to_dataclass(delta_key_to_value, delta_view_type)
        # delta_view = left_entity_type[DeltaView]
        return delta_view
