from dataclasses import dataclass
from typing import Type

from ..entity_view.viewable_entity import ViewableEntity
from extended_logger import get_logger
from .delta_view_renderer import DeltaViewRenderer, EntityRenderPattern
from ..translation_loader import TranslationLoader

logger = get_logger(__name__)
DEFAULT_PATTERN_NAME = "_default"


class DeltaViewWeblateRenderer(DeltaViewRenderer):
    @dataclass
    class Context:
        project_name: str
        translation_loader: TranslationLoader

    @dataclass
    class Config:
        weblate_component_name: str = 'delta_view_chat_renderer'
        weblate_translation_language: str = 'ru'

    def __init__(self, config: Config, context: Context) -> None:
        super().__init__()
        logger.debug(f"init DeltaViewWeblateRenderer: config={config}, context={context}")
        self.config = config
        self.context = context

    async def load_templates_for_entity_types(self, entity_types: list[Type[ViewableEntity]]) -> None:
        logger.info(f"load_templates_for_entity_types({entity_types})")
        entity_name_to_patterns = await self.context.translation_loader.get_translations_file_content(
            project=self.context.project_name.lower(), component=self.config.weblate_component_name,
            language=self.config.weblate_translation_language
        )
        logger.debug(
            f"loaded translation project={self.context.project_name.lower()}, "
            f"component={self.config.weblate_component_name}, language={self.config.weblate_translation_language}"
        )
        print(f"{entity_name_to_patterns=}")

        default_pattern = entity_name_to_patterns[DEFAULT_PATTERN_NAME]
        for type_ in entity_types:
            entity_pattern = {**default_pattern, **entity_name_to_patterns.get(type_.get_table_name(), entity_name_to_patterns["_default"])}
            pattern = EntityRenderPattern(**entity_pattern)
            self.register_pattern_for_entity_type(entity_type=type_, pattern=pattern)
            logger.debug(f"loaded template for {type_.__name__}")
