#  Copyright (C) 2023
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Revva Konstantin <k.revva@abm-jsc.ru>
import time
from abc import ABC
from dataclasses import dataclass, field
from typing import Optional, Type
import json
from http_tools.http_server_connector import HttpServerConnector
from init_helpers.dict_to_dataclass import dict_to_dataclass
from init_helpers.custom_json import custom_dumps
from time_tools import Timerange


@dataclass(kw_only=True, frozen=True)
class DetectorSettings(ABC):
    rtsp_url: str
    created_at: float = field(default_factory=time.time)


class DetectorHttpConnector:
    @dataclass
    class Context(HttpServerConnector.Context):
        project_name: str

    Config = HttpServerConnector.Config

    def __init__(self,
                 config: Config,
                 context: Context,
                 detector_settings_type: Type[DetectorSettings]) -> None:
        self.config = config
        self.context = context
        self._connector = HttpServerConnector(config, context)
        self.detector_settings_type = detector_settings_type

    async def get_settings(self, rtsp_urls: list[str],
                           token: Optional[str] = None, is_inner_request: bool = False) -> list[DetectorSettings]:
        payload = {"rtsp_urls": rtsp_urls}

        answer = await self._auth_post_json("/settings/get",
                                            payload=payload,
                                            token=token,
                                            is_inner_request=is_inner_request,
                                            )

        return [dict_to_dataclass(settings, self.detector_settings_type) for settings in answer["result"]]

    async def set_settings(self, detector_settings: DetectorSettings,
                           token: Optional[str] = None, is_inner_request: bool = False) -> DetectorSettings:

        payload = {"settings": json.loads(custom_dumps(detector_settings))}

        answer = await self._auth_post_json("/settings/set",
                                            payload=payload,
                                            token=token,
                                            is_inner_request=is_inner_request,
                                            )

        return dict_to_dataclass(answer["result"], self.detector_settings_type)

    async def autotune_settings(self, rtsp_url: str, miss_decision_timerange: Timerange,
                                token: Optional[str] = None, is_inner_request: bool = False) -> DetectorSettings:

        payload = {
            "rtsp_url": rtsp_url,
            "miss_decision_timerange": json.loads(custom_dumps(miss_decision_timerange))
        }

        answer = await self._auth_post_json("/settings/autotune",
                                            payload=payload,
                                            token=token,
                                            is_inner_request=is_inner_request,
                                            )

        return dict_to_dataclass(answer["result"], self.detector_settings_type)

    async def _auth_post_json(self, url: str, payload: dict, headers: dict | None = None,
                              token: str | None = None, is_inner_request: bool = False) -> dict:
        headers = headers if headers is not None else {}
        assert not (token and is_inner_request), "only one of 'token'/'is_inner_request' is allowed"
        if token:
            payload["token"] = token
        elif is_inner_request:
            headers["server_name"] = self.context.project_name
        return await self._connector.post_json(url,
                                               payload=payload,
                                               headers=headers)

