#  Copyright (C) 2024
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mike Orlov <m.orlov@abm-jsc.ru>

import sqlalchemy as sa

from .columns import create_columns, drop_columns, set_not_null
from .migrate_data import update_table_data
from .trigger import create_triggers, drop_triggers


def upgrade(
        table_name_to_old_columns_to_migration_expression: dict[str, dict[sa.Column, sa.ColumnElement | None]],
        table_name_to_new_columns_to_migration_expression: dict[str, dict[sa.Column, sa.ColumnElement | None]],
        revision: str
) -> None:
    tables = table_name_to_old_columns_to_migration_expression | table_name_to_new_columns_to_migration_expression
    for table_name in tables:
        old_columns_to_migration_expression = table_name_to_old_columns_to_migration_expression.get(table_name, {})
        new_columns_to_migration_expression = table_name_to_new_columns_to_migration_expression.get(table_name, {})
        create_columns(table_name, new_columns_to_migration_expression.keys(), force_nullable=True)
        create_triggers(table_name, old_columns_to_migration_expression, new_columns_to_migration_expression, revision)
        update_table_data(table_name, new_columns_to_migration_expression)
        set_not_null(table_name, [col for col in old_columns_to_migration_expression if col.nullable is False])


def finalize(
        table_name_to_old_columns_to_migration_expression: dict[str, dict[sa.Column, sa.ColumnElement | None]],
        table_name_to_new_columns_to_migration_expression: dict[str, dict[sa.Column, sa.ColumnElement | None]],
        revision: str
) -> None:
    tables = table_name_to_old_columns_to_migration_expression | table_name_to_new_columns_to_migration_expression
    for table_name in tables:
        old_columns_to_migration_expression = table_name_to_old_columns_to_migration_expression.get(table_name, {})
        drop_triggers(table_name, revision)
        drop_columns(table_name, old_columns_to_migration_expression.keys())


def undo_finalize(
        table_name_to_old_columns_to_migration_expression: dict[str, dict[sa.Column, sa.ColumnElement | None]],
        table_name_to_new_columns_to_migration_expression: dict[str, dict[sa.Column, sa.ColumnElement | None]],
        revision: str
) -> None:
    tables = table_name_to_old_columns_to_migration_expression | table_name_to_new_columns_to_migration_expression
    for table_name in tables:
        old_columns_to_migration_expression = table_name_to_old_columns_to_migration_expression.get(table_name, {})
        new_columns_to_migration_expression = table_name_to_new_columns_to_migration_expression.get(table_name, {})

        create_columns(table_name, old_columns_to_migration_expression.keys(), force_nullable=True)
        create_triggers(table_name, old_columns_to_migration_expression, new_columns_to_migration_expression, revision)
        update_table_data(table_name, old_columns_to_migration_expression)
        set_not_null(table_name, [col for col in old_columns_to_migration_expression if col.nullable is False])


def downgrade(
        table_name_to_old_columns_to_migration_expression: dict[str, dict[sa.Column, sa.ColumnElement | None]],
        table_name_to_new_columns_to_migration_expression: dict[str, dict[sa.Column, sa.ColumnElement | None]],
        revision: str
) -> None:
    tables = table_name_to_old_columns_to_migration_expression | table_name_to_new_columns_to_migration_expression
    for table_name in tables:
        new_columns_to_migration_expression = table_name_to_new_columns_to_migration_expression.get(table_name, {})
        drop_triggers(table_name, revision)
        drop_columns(table_name, new_columns_to_migration_expression.keys())
