from dataclasses import dataclass, field
from typing import ClassVar

from aiohttp import web
from http_tools import Answer, HttpStatusCode, HttpServer
from http_tools.mime_types import ContentType
from prometheus_client import generate_latest
from prometheus_client.utils import INF

from .collectors.background_tasks_metrics_collector import BackgroundTasksMetricsCollector
from .collectors.incoming_http_requests_metrics_collector import IncomingHttpRequestsMetricsCollector
from .collectors.outgoing_http_requests_metrics_collector import OutgoingHttpRequestsMetricsCollector
from .monitored_components.monitored_client_session import MonitoredClientSession
from .metrics_registry import MetricsRegistry, METRICS_REGISTRY


class MetricsGateway:
    METRICS_ENDPOINT: ClassVar[str] = "/metrics"

    @dataclass
    class Config:
        http_request_latency_buckets: list[float] = field(
            default_factory=lambda: [.05, .1, .25, .5, .75, 1., 2.5, 5, 7.5, 10., 30., 60., 120., INF],
        )
        background_tasks_duration_buckets: list[float] = field(
            default_factory=lambda: [.05, .1, .25, .5, .75, 1., 2.5, 5, 7.5, 10., 30., 60., 120., INF]
        )

    @dataclass
    class Context:
        http_server: HttpServer

    def __init__(self, config: Config, context: Context, registry: MetricsRegistry | None = None) -> None:
        self.config = config
        self.context = context
        self._registry = METRICS_REGISTRY if registry is None else registry

        self.incoming_http_requests_metrics_collector = IncomingHttpRequestsMetricsCollector(
            config=IncomingHttpRequestsMetricsCollector.Config(
                exclude_paths=[self.METRICS_ENDPOINT],
                latency_buckets=self.config.http_request_latency_buckets,
            ),
            context=IncomingHttpRequestsMetricsCollector.Context(self._registry),
        )
        self.outgoing_http_requests_metrics_collector = OutgoingHttpRequestsMetricsCollector(
            config=OutgoingHttpRequestsMetricsCollector.Config(self.config.http_request_latency_buckets),
            context=OutgoingHttpRequestsMetricsCollector.Context(self._registry),
        )
        self.background_tasks_metrics_collector = BackgroundTasksMetricsCollector(
            config=BackgroundTasksMetricsCollector.Config(self.config.background_tasks_duration_buckets),
            context=BackgroundTasksMetricsCollector.Context(self._registry),
        )

        self.ensure_counter = self._registry.ensure_counter
        self.ensure_gauge = self._registry.ensure_gauge
        self.ensure_histogram = self._registry.ensure_histogram
        self.ensure_summary = self._registry.ensure_summary

        self.context.http_server.register_handler(
            self.METRICS_ENDPOINT,
            lambda _: Answer(generate_latest(self._registry), status=HttpStatusCode.OK, content_type=ContentType.Text),
        )

    def get_monitored_client_session(self) -> MonitoredClientSession:
        return MonitoredClientSession(self.outgoing_http_requests_metrics_collector)

    def get_incoming_http_requests_middleware(self) -> web.middleware:
        return self.incoming_http_requests_metrics_collector.create_middleware()
