import datetime
from dataclasses import dataclass
from typing import Self, ClassVar

from .timestamp_sec import TimestampSec


@dataclass(slots=True)
class Timerange:
    start: TimestampSec | None
    end: TimestampSec | None
    DEFAULT_TIMERANGE_FORMAT: ClassVar[str] = '{start}/{end}'

    def __init__(
            self,
            start: TimestampSec | datetime.datetime | float | int | None,
            end: TimestampSec | datetime.datetime | float | int | None
    ) -> None:
        if start is not None and end is not None and start > end:
            raise ValueError(f"Bad timerange: start({start}) > end({end})")
        # pycharm analyser errors
        # noinspection PyTypeChecker
        self.start = None if start is None else TimestampSec(start)
        # noinspection PyTypeChecker
        self.end = None if end is None else TimestampSec(end)

    def to_datetime_str(
            self, timezone: datetime.timezone | None = None,
            datetime_format: str | None = None,
            timerange_format: str | None = None,
    ) -> str:
        timerange_format = timerange_format or self.DEFAULT_TIMERANGE_FORMAT
        start = "..." if self.start is None else self.start.to_datetime_str(timezone=timezone, format_=datetime_format)
        end = "..." if self.end is None else self.end.to_datetime_str(timezone=timezone, format_=datetime_format)
        return timerange_format.format(start=start, end=end)

    def to_date_str(
            self, timezone: datetime.timezone | None = None,
            date_format: str | None = None,
            timerange_format: str | None = None,
    ) -> str:
        timerange_format = timerange_format or self.DEFAULT_TIMERANGE_FORMAT
        start = "..." if self.start is None else self.start.to_date_str(timezone=timezone, format_=date_format)
        end = "..." if self.end is None else self.end.to_date_str(timezone=timezone, format_=date_format)
        return timerange_format.format(start=start, end=end)

    def is_closed(self) -> bool:
        return self.start is not None and self.end is not None

    def duration(self) -> datetime.timedelta | None:
        if self.is_closed():
            # pycharm analyser cannot read overload of __sub__ method
            # noinspection PyTypeChecker
            return self.end - self.start

    def __bool__(self) -> bool:
        return self.start is None or self.start != self.end

    def intersection(self, timerange: Self) -> Self | None:
        if timerange.start is None:
            start = self.start
        elif self.start is None:
            start = timerange.start
        else:
            start = max(self.start, timerange.start)

        if timerange.end is None:
            end = self.end
        elif self.end is None:
            end = timerange.end
        else:
            end = min(self.end, timerange.end)

        if start and end and start > end:
            return None
        return Timerange(start, end)
