#  Copyright (C) 2025
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mike Orlov <m.orlov@abm-jsc.ru>
import functools
from dataclasses import dataclass
from typing import ClassVar, Mapping

from frozendict import frozendict
from init_helpers import Jsonable

from openapi_tools.spec.spec_resource import SpecResource, SpecRef
from .schema_format import IntegerSchemaFormat
from .type_schema import TypeSchema


@dataclass(frozen=True, slots=True)
class IntegerSchema(TypeSchema):
    type_: ClassVar[str] = "integer"
    format_: IntegerSchemaFormat | None = None
    exclusive_minimum: int | None = None
    exclusive_maximum: int | None = None

    def get_spec_dict(self, dependency_to_ref: Mapping['SpecResource', SpecRef | dict]) -> frozendict[str, Jsonable]:
        result = {}
        if self.exclusive_minimum is not None:
            result |= {'exclusiveMinimum': self.exclusive_minimum}
        if self.exclusive_maximum is not None:
            result |= {'exclusiveMaximum': self.exclusive_maximum}
        return frozendict(TypeSchema.get_spec_dict(self, dependency_to_ref) | result)

    @functools.cache
    def _get_repr_parts(self) -> tuple[str, ...]:
        parts = list(TypeSchema._get_repr_parts(self))
        parts += [f'exclusive_minimum={self.exclusive_minimum!r}'] if self.exclusive_minimum else []
        parts += [f'exclusive_maximum={self.exclusive_maximum!r}'] if self.exclusive_maximum else []
        return tuple(parts)

    def __repr__(self):
        return f'{self.__class__.__name__}({", ".join(self._get_repr_parts())})'

    __str__ = __repr__
