#  Copyright (C) 2023
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasiliev Ivan <i.vasiliev@technokert.ru>


import logging
import warnings
from dataclasses import dataclass
from typing import Optional

from http_tools import AbmServiceConnector
from init_helpers.dict_to_dataclass import dict_to_dataclass


logger = logging.getLogger(__name__)


@dataclass
class FileInfo:
    key: str
    md5: str
    sha512: str
    size: int


@dataclass
class FileLink:
    id: str
    filename: str
    extension: str | None
    file_info: FileInfo

    @property
    def size(self) -> int:
        warnings.warn("'size' is deprecated, replace it with 'file_info.size'", DeprecationWarning)
        return self.file_info.size

    @property
    def full_file_name(self) -> str:
        warnings.warn("'full_file_name' is deprecated, replace it with 'filename'", DeprecationWarning)
        return self.filename

    @property
    def file_extension(self) -> str | None:
        warnings.warn("'file_extension' is deprecated, replace it with 'extension'", DeprecationWarning)
        return self.extension

    @property
    def quoted_full_file_name(self) -> str:
        warnings.warn("'quoted_full_file_name' is deprecated, was removed", DeprecationWarning)
        return self.filename

    @property
    def content_type(self) -> str:
        warnings.warn("'content_type' is deprecated, was removed", DeprecationWarning)
        return ''

    @property
    def thumbnail(self) -> None:
        warnings.warn("'thumbnail' is deprecated, was removed", DeprecationWarning)
        return None


FileDescriptor = FileLink  # backward compatibility


class FileServerConnector:
    Config = AbmServiceConnector.Config
    Context = AbmServiceConnector.Context

    def __init__(self, config: Config, context: Context):
        self._connector = AbmServiceConnector(config, context)

    async def get_file(self, file_id: str, range_: Optional[str] = None) -> bytes:
        args = {"id": file_id}
        headers = self._construct_headers()
        if range_ is not None:
            headers["Range"] = range_
        return await self._connector.get("/file/get", args, headers=headers)

    async def get_file_info(self, file_id: str) -> FileLink:
        args = {"id": file_id}
        answer = await self._connector.get("/file/info", args, headers=self._construct_headers())
        return dict_to_dataclass(answer, FileLink)

    async def add_file(self, file: bytes, filename: str, is_thumbnail_required: bool = False) -> FileLink:
        args = {"filename": filename, "is_thumbnail_required": is_thumbnail_required}
        answer = await self._connector.post("/file/add", payload=file, url_query=args,
                                            headers=self._construct_headers())
        return dict_to_dataclass(answer, FileLink)

    async def delete_file(self, file_id: str) -> FileLink:
        args = {"id": file_id}
        answer = await self._connector.post("/file/delete", payload=None,
                                            url_query=args, headers=self._construct_headers())
        return dict_to_dataclass(answer, FileLink)

    def _construct_headers(self) -> dict[str, str]:
        headers = {"Connection": "Keep-Alive"}
        return headers
