#  Copyright (C) 2022
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mike Orlov <m.orlov@abm-jsc.ru>
#
import dataclasses
import functools
from typing import Type, List, Optional, Any, Union, Iterable

from sqlalchemy import Column
from sqlalchemy.orm import InstrumentedAttribute

from extended_logger import get_logger
from .select_atoms import ERoot, EAlias, EFunctionLen, EColumn, ELiteral, EOrder, \
    EFunction, EFunctionGT, EFunctionEQ, EFunctionIncluded, Node, EFunctionStringContains, EFunctionCastToStr, \
    EFunctionIsNull, EFunctionNotNull, EFunctionAnd, EFunctionOr, EFunctionLT, EFunctionGE, EFunctionLE
from ..components import EntityProxy, AttributeProxy

from ..viewable_entity import ViewableEntity

logger = get_logger(__name__)


def _auto_cast(value, e_root_allowed: bool = True):
    if isinstance(value, (bool, int, float, list, set, str)):
        return ELiteral(value)
    if isinstance(value, (InstrumentedAttribute, Column)):
        return EColumn(value.key)
    if isinstance(value, ERoot):
        return value if e_root_allowed else EColumn(value.alias)
    if isinstance(value, (EColumn, ELiteral, EFunction, ERoot, EAlias)):
        return value
    if isinstance(value, EntityProxy):
        return value
    if isinstance(value, AttributeProxy):
        return value
    logger.error(f'_auto_cast>- unexpected value: %s(%s)', value, type(value))
    assert False


_auto_cast_strict = functools.partial(_auto_cast, e_root_allowed=False)


class FunctionCollection:
    @staticmethod
    def greater(left, right) -> EFunctionGT:
        return EFunctionGT([_auto_cast_strict(left), _auto_cast_strict(right)])

    @staticmethod
    def greater_or_equal(left, right) -> EFunctionGE:
        return EFunctionGE([_auto_cast_strict(left), _auto_cast_strict(right)])

    @staticmethod
    def less(left, right) -> EFunctionLT:
        return EFunctionLT([_auto_cast_strict(left), _auto_cast_strict(right)])

    @staticmethod
    def less_or_equal(left, right) -> EFunctionLE:
        return EFunctionLE([_auto_cast_strict(left), _auto_cast_strict(right)])

    @staticmethod
    def equal(left, right) -> EFunctionEQ:
        return EFunctionEQ([_auto_cast_strict(left), _auto_cast_strict(right)])

    @staticmethod
    def len(value) -> EFunctionLen:
        return EFunctionLen([_auto_cast_strict(value)])

    @staticmethod
    def included(what, into) -> EFunctionIncluded:
        return EFunctionIncluded([_auto_cast_strict(what), _auto_cast_strict(into)])

    @staticmethod
    def string_contains(what, into) -> EFunctionStringContains:
        return EFunctionStringContains([_auto_cast_strict(what), _auto_cast_strict(into)])

    @staticmethod
    def to_str(what) -> EFunctionCastToStr:
        return EFunctionCastToStr([_auto_cast_strict(what)])

    @staticmethod
    def is_null(what) -> EFunctionIsNull:
        return EFunctionIsNull([_auto_cast_strict(what)])

    @staticmethod
    def not_null(what) -> EFunctionNotNull:
        return EFunctionNotNull([_auto_cast_strict(what)])

    @staticmethod
    def or_(*what) -> EFunctionOr:
        return EFunctionOr([_auto_cast_strict(val) for val in what])

    @staticmethod
    def and_(*what) -> EFunctionAnd:
        return EFunctionAnd([_auto_cast_strict(val) for val in what])


op = FunctionCollection


def order(column, asc: bool = True, nulls_last: Optional[bool] = None) -> EOrder:
    if not isinstance(column, (EColumn, EFunction)):
        column = _auto_cast(column)
    result = EOrder(column, asc=asc, nulls_last=nulls_last)
    return result


def _query(entity: Union[Type[ViewableEntity], str], alias: Optional[str] = None, attrs: Optional[Iterable[Any]] = None,
           vars: Optional[Iterable[Any]] = None, filter: Optional[Iterable[Any]] = None,
           order: Optional[Iterable[EOrder]] = None, limit: Optional[int] = None,
           offset: Optional[int] = None) -> ERoot:
    attrs = attrs or []
    vars = vars or []
    filter = filter or []
    order = [] if order is None else list(order)

    e_attrs = list(map(_auto_cast, attrs))
    e_vars = list(map(_auto_cast, vars))
    e_filter = list(map(_auto_cast, filter))
    result = ERoot(
        entity, alias=alias, attrs=e_attrs, vars=e_vars, filter=e_filter, order=order, limit=limit, offset=offset
    )
    return result


# noinspection PyTypeChecker
def query(entity: Type[ViewableEntity], alias: Optional[str] = None, attrs: Optional[Iterable[Any]] = None,
          vars: Optional[Iterable[Any]] = None, filters: Optional[Iterable[Any]] = None,
          orders: Optional[Iterable[EOrder]] = None,
          limit: Optional[int] = None, offset: Optional[int] = None) -> ERoot:
    return _query(
        entity=entity, alias=alias, attrs=attrs, vars=vars, filter=filters, order=orders, limit=limit, offset=offset
    )


def alias(value: Node, alias: str) -> EAlias:
    return EAlias(_auto_cast_strict(value), alias)


def subquery(relation, alias: Optional[str] = None, attrs: Optional[Iterable[Any]] = None,
             vars: Optional[Iterable[Any]] = None, filter: Optional[Iterable[Any]] = None,
             order: Optional[Iterable[EOrder]] = None):
    key = relation.key
    alias = relation.key if alias is None else alias
    return _query(entity=key, alias=alias, attrs=attrs, vars=vars, filter=filter, order=order)

@dataclasses.dataclass
class Rule:
    object = ...
    action = ...
    subject = ...
    limited = ...

User = ...

# class Action(

Rule()
rulebook = ...


class Rulebook:
    def apply_rules(self, query: ERoot, user_id):
        pass


# # noinspection PyCallingNonCallable
# q = Selector(
#     Agent,
#     attrs=[
#         Agent.id,
#         Agent.screens(attrs=[
#             Screen.id, Screen.key,
#             Screen.playlists(attrs=[ScreenWindow.layout_x, ScreenWindow.layout_y])
#         ])],
#     vars=[
#         active_screens := Agent.screens(
#             attrs=[Screen.id, Screen.key, playlists := Screen.playlists()],
#             filter=[playlists.len().gt(0)]
#         ).label("active_screens")
#     ],
#     filter=[
#         active_screens.len().gt(0)
#     ]
# )
# Q = ERoot(
#     Agent,
#     attrs=[
#         EAlias(EFunctionLen([EColumn("screens")]), alias="screens_amount"),
#         EColumn("id"), EAlias(EColumn("id"), alias="id2"), EAlias(ELiteral(1), alias="one"), EColumn("key"),
#         ERoot(
#             "screens",
#             attrs=[
#                 EColumn("id"), EAlias(EColumn("key"), alias="screen_key"), EAlias(ELiteral(2), alias="two"),
#                 ERoot("playlists", attrs=[EColumn("layout_x"), EColumn("layout_y")])
#             ],
#             # filter=[EFunctionLE([EColumn("id"), ELiteral(10)])]
#             # order=[EOrder(EColumn("id"), asc=False)]
#         ),
#         # ERoot(
#         #     "stages",
#         #     attrs=[EColumn("id"), EColumn("start_at"), EColumn("end_at")],
#         #     order=[EOrder(EColumn("start_at"), asc=False)],
#         #     alias="active_stages2",
#         # ),
#     ],
#     vars=[
#         ERoot(
#             "screens", attrs=[ERoot("playlists")],
#             alias="active_screens",
#             filter=[
#                 EFunctionGE([EFunctionLen([EColumn("playlists")]), ELiteral(1)])
#             ],
#         )
#     ],
#     filter=[
#         EFunctionGE([EFunctionLen([EColumn("active_screens")]), ELiteral(1)])
#     ],
#     order=[EOrder(EFunctionLen([EColumn("screens")]), asc=True)]
# )
