#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>

import logging
import os
import shutil
from concurrent.futures import ThreadPoolExecutor
from dataclasses import dataclass
from typing import Optional

import aiofiles
from aiofiles.os import remove
from aiofiles.ospath import exists

from ..abstract_file_storage import AbstractFileStorage
from ..exceptions import FileAlreadyExists, PathDoesNotExists, InsecurePath

logger = logging.getLogger(__name__)


@dataclass
class MemoryState:
    total_gb: float
    used_gb: float
    free_gb: float
    occupancy_percentage: float


class DiscFileStorage(AbstractFileStorage):
    @dataclass
    class Config:
        location: str
        max_workers: Optional[int] = None

    Context = AbstractFileStorage.Context

    def __init__(self, config: Config, context: Context) -> None:
        super().__init__(context)
        self.config = config
        self._thread_pool_executor = ThreadPoolExecutor(max_workers=self.config.max_workers)
        logger.info(f"{type(self).__name__} inited. Location: {self.config.location}")

    async def _save(self, content: bytes, relative_path: str, name: str, allow_rewrite: bool = False) -> None:
        dir_path = f"{self.config.location}/{relative_path}"
        if not await aiofiles.ospath.exists(dir_path, executor=self._thread_pool_executor):
            logger.info(f"Path = {dir_path} does not exists. Will be created")
            await aiofiles.os.makedirs(dir_path)

        full_path = f"{dir_path}/{name}"
        logger.info(f"Full path = {full_path}")
        if await aiofiles.ospath.exists(full_path, executor=self._thread_pool_executor) and not allow_rewrite:
            raise FileAlreadyExists(full_path)

        logger.info(f"Saving file along the path: {full_path}")
        async with aiofiles.open(full_path, "wb", executor=self._thread_pool_executor) as file:
            await file.write(content)

    async def _load(self, relative_path: str, name: str, offset: int = 0, size: int = -1) -> bytes:
        dir_path = f"{self.config.location}/{relative_path}"
        full_path = f"{dir_path}/{name}"
        if not await aiofiles.ospath.exists(full_path, executor=self._thread_pool_executor):
            raise PathDoesNotExists(full_path)

        if not os.path.realpath(full_path).startswith(os.path.realpath(dir_path)):
            raise InsecurePath(full_path)

        logger.info(f"Loading file along the path: {full_path}")
        async with aiofiles.open(full_path, "rb", executor=self._thread_pool_executor) as file:
            await file.seek(offset)
            content = await file.read(size)
        return content

    async def _delete(self, relative_path: str, name: str) -> None:
        dir_path = f"{self.config.location}/{relative_path}"
        full_path = f"{dir_path}/{name}"

        if not await aiofiles.ospath.exists(full_path, executor=self._thread_pool_executor):
            raise PathDoesNotExists(full_path)

        logger.info(f"Deleting file along the path: {full_path}")
        await aiofiles.os.remove(full_path, executor=self._thread_pool_executor)
        if not await aiofiles.os.listdir(dir_path, executor=self._thread_pool_executor):
            await aiofiles.os.rmdir(dir_path, executor=self._thread_pool_executor)

    async def _check_file_existence(self, relative_path: str, name: str) -> bool:
        full_path = f"{self.config.location}/{relative_path}/{name}"
        return await aiofiles.ospath.exists(full_path, executor=self._thread_pool_executor)

    def get_memory_state(self) -> MemoryState:
        result = shutil.disk_usage(self.config.location)
        gb = 10**9
        return MemoryState(
            total_gb=float(f"{result.total/gb:.2f}"),
            used_gb=float(f"{result.used/gb:.2f}"),
            free_gb=float(f"{result.free/gb:.2f}"),
            occupancy_percentage=float(f"{(result.used / result.total) * 100:.2f}"),
        )
