#  Copyright (C) 2025
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mike Orlov <m.orlov@abm-jsc.ru>
import functools
from dataclasses import dataclass, field
from typing import Mapping, Iterable

from frozendict import frozendict
from init_helpers import Jsonable

from openapi_tools.spec import SpecResource
from ..scheme import SecurityScheme


@dataclass(frozen=True, slots=True)
class SecurityRequirement(SpecResource):
    scheme_to_scopes: Mapping[SecurityScheme, frozenset[str]] = field(default_factory=frozendict)

    def __init__(self, scheme_to_scopes: Mapping[SecurityScheme, Iterable[str]] | None = None):
        object.__setattr__(
            self, 'scheme_to_scopes',
            frozendict({scheme: frozenset(scopes) for scheme, scopes in scheme_to_scopes.items()}))

    def get_spec_dependencies(self) -> frozenset['SpecResource']:
        return frozenset(self.scheme_to_scopes.keys())

    def get_spec_dict(self, dependency_to_ref: Mapping['SpecResource', str]) -> frozendict[str, Jsonable]:
        # noinspection PyTypeChecker
        return frozendict({dependency_to_ref[scheme]: scopes for scheme, scopes in self.scheme_to_scopes.items()})

    def _repr(self) -> list[str]:
        parts = []
        if self.scheme_to_scopes:
            parts.append(f'scheme_to_scopes={ {scheme: tuple(scopes) for scheme, scopes in self.scheme_to_scopes.items()} }')
        return parts

    def __repr__(self):
        return f'{self.__class__.__name__}({", ".join(self._repr())})'

    __str__ = __repr__
