#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mike Orlov <m.orlov@abm-jsc.ru>
#
import asyncio
from logging import getLogger
from typing import Any

from .recursive import get_recursive

logger = getLogger(__name__)


class AsyncInitable:
    def __init__(self):
        self.async_init_scheduled__event: asyncio.Event = asyncio.Event()  # called async_init
        self._tasks_before_async_init: list[asyncio.Task] = []
        self.async_init_started__event: asyncio.Event = asyncio.Event()  # passed wait
        self.async_init_finished__event: asyncio.Event = asyncio.Event()  # done own _async_init

    def wait_before_async_init_for(self, task: asyncio.Task) -> None:
        self._tasks_before_async_init.append(task)

    async def async_init(self) -> None:
        if self.async_init_scheduled__event.is_set():
            logger.warning(f"async_start called extra time on {self}")
            return
        self.async_init_scheduled__event.set()
        if self._tasks_before_async_init:
            logger.debug(f"{self} before starting own _async_init is waiting for {self._tasks_before_async_init}")
            await asyncio.gather(*self._tasks_before_async_init)
        self.async_init_started__event.set()
        logger.debug(f"{self} started _async_init")
        await self._async_init()
        logger.debug(f"{self} finished _async_init")
        self.async_init_finished__event.set()

    async def _async_init(self) -> None:
        pass

    @staticmethod
    async def async_init_recursive(target: Any) -> None:
        async_initables = await get_recursive(target, condition=lambda x: isinstance(x, AsyncInitable))
        await asyncio.gather(*[x.async_init() for x in async_initables if not x.async_init_scheduled__event.is_set()])
