#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>, Alexander Medvedev <a.medvedev@abm-jsc.ru>
import asyncio
from concurrent.futures import ProcessPoolExecutor
from typing import Iterable

from .abstract_thumbnail_processor import AbstractThumbnailProcessor


class ThumbnailProvider:
    class Config:
        process_pool_size: int = 5

    def __init__(self, config: Config, *processors: AbstractThumbnailProcessor) -> None:
        self.config = config
        self._process_pool = ProcessPoolExecutor(max_workers=self.config.process_pool_size)
        self._file_extension_to_thumbnail_processor: dict[str, AbstractThumbnailProcessor] = {}
        for processor in processors:
            self.register(processor.processable_extensions, processor)

    def register(self, file_extensions: Iterable[str], thumbnail_processor: AbstractThumbnailProcessor) -> None:
        for file_extension in file_extensions:
            self._file_extension_to_thumbnail_processor[file_extension] = thumbnail_processor

    def get(self, file_extension: str) -> AbstractThumbnailProcessor | None:
        return self._file_extension_to_thumbnail_processor.get(file_extension)

    async def provide(self, file_extension: str, content: bytes) -> bytes | None:
        if processor := self._file_extension_to_thumbnail_processor.get(file_extension):
            return await asyncio.get_event_loop().run_in_executor(self._process_pool, processor.process, content)