#  Copyright (C) 2024
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mike Orlov <m.orlov@abm-jsc.ru>
from dataclasses import dataclass

from file_storage.abstract_file_storage import AbstractFileStorage

from .database import Database
from .entities.link_info import LinkInfo
from .extra import silent
from .file_keeper import FileKeeper
from .link_keeper import LinkKeeper
from .memory_zipper import MemoryZipper
from .thumbnail.thumbnail_provider import ThumbnailProvider


class Controller:
    @dataclass
    class Config:
        pass

    @dataclass
    class Context:
        database: Database
        # zipper: MemoryZipper
        file_storage: AbstractFileStorage
        # image_thumbnail_processor: ImageThumbnailProcessor
        thumbnail_provider: ThumbnailProvider
        # file_database_facade: FileDatabaseFacade
        # thumbnail_provider: ThumbnailProvider

    def __init__(self, config: Config, context: Context) -> None:
        self.config = config
        self.context = context

        self.file_keeper = FileKeeper(FileKeeper.Context(self.context.database, self.context.file_storage))
        self.link_keeper = LinkKeeper(LinkKeeper.Context(self.file_keeper, self.context.database))
        self.thumbnail_keeper = LinkKeeper(LinkKeeper.Context(self.file_keeper, self.context.database))
        self.zipper = MemoryZipper(MemoryZipper.Context(self.link_keeper, self.context.database))
        # self.thumbnail_provider = ThumbnailProvider(self.context.image_thumbnail_processor)

    def _publish(self):
        ...

    async def upload(self, content: bytes, filename: str, is_thumbnail_required: bool = False) -> LinkInfo:
        async with self.context.database.ensure_session() as session:
            link_info = await self.link_keeper.upload(content, filename, session=session)
            if is_thumbnail_required:
                if thumbnail := await self.context.thumbnail_provider.provide(link_info.extension, link_info):
                    await self.thumbnail_keeper.upload(thumbnail, filename, link_id=link_info.id, session=session)
        return link_info

    async def get(self, link_id: str, byte_range: range | None = None) -> tuple[LinkInfo, bytes]:
        link_info, file_content = await self.link_keeper.get(link_id, byte_range=byte_range)
        return link_info, file_content

    async def get_thumbnail(self, link_id: str, byte_range: range | None = None) -> tuple[LinkInfo, bytes]:
        link_info, file_content = await self.thumbnail_keeper.get(link_id, byte_range=byte_range)
        return link_info, file_content

    async def head(self, link_id: str) -> LinkInfo:
        return await self.link_keeper.head(link_id)

    async def delete(self, link_id: str) -> LinkInfo:
        async with self.context.database.ensure_session() as session:
            result = await self.link_keeper.delete(link_id, session)
            await silent(self.thumbnail_keeper.delete(link_id, session))
        return result

    async def zip_files(self, link_ids: list[str]) -> tuple[LinkInfo, bytes]:
        return await self.zipper.zip_files(link_ids)
