#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>, Alexander Medvedev <a.medvedev@abm-jsc.ru>

from functools import wraps
from typing import Union, Callable, Awaitable

from file_storage.exceptions import PathDoesNotExists
from http_tools import ErrorAnswer, Answer, HttpStatusCode
from http_tools.request import IncomingRequest
from sqlalchemy.exc import SQLAlchemyError
from botocore.exceptions import ConnectionError

from ..file_database_facade import FileMetadataDoesNotExists


Handler = Union[Callable[[IncomingRequest], Awaitable[Answer]], Callable[[IncomingRequest], Answer]]


def error_middleware(handler: Handler):
    @wraps(handler)
    async def wrapper(*args, **kwargs) -> Answer:
        try:
            return await handler(*args, **kwargs)
        except (FileMetadataDoesNotExists, PathDoesNotExists) as er:
            return ErrorAnswer(repr(er), status=HttpStatusCode.NotFound)
        except (ConnectionError, SQLAlchemyError) as er:
            return ErrorAnswer(repr(er), status=HttpStatusCode.BadGateway)
    return wrapper
