from dataclasses import asdict, dataclass
from logging import Logger
import aiohttp
from aiokafka import AIOKafkaProducer

from async_tools import AsyncOnStart, AsyncOnStop

from notification_server_connector.models.notification import Notification
from notification_server_connector.utils.abstract_encoder import AbstractEncoder


logger = Logger(__name__)


class NotificationConnector(AsyncOnStart, AsyncOnStop):
    @dataclass
    class Config:
        url: str
        address: str
        max_request_size: int
        topic: str

    @dataclass
    class Context:
        encoder: AbstractEncoder

    _config: Config
    _producer: AIOKafkaProducer
    _context: Context

    def __init__(self, config: Config, context: Context) -> None:
        self._config = config
        self._context = context
        self._http_client = aiohttp.ClientSession()

    async def _on_start(self) -> None:
        self._producer = AIOKafkaProducer(
            bootstrap_servers=self._config.address,
            max_request_size=self._config.max_request_size,
        )
        await self._producer.start()

    async def _on_stop(self) -> None:
        await self._producer.stop()

    async def send(self, notification: Notification) -> bool:
        logger.debug(f'write to kafka channel "{self._config.topic}": {notification}')
        payload = self._context.encoder.encode(asdict(notification))
        try:
            result = await self._producer.send(self._config.topic, value=payload)
            logger.debug(f'write to kafka channel "{self._config.topic}" done, result {result}')
            return True
        except Exception as exc:
            logger.error(f'write to kafka channel "{self._config.topic}" failed, error: {exc}')
            return False

    async def mark_notification_as_read_internal(self, source_type: str, notification_ids: list[int]) -> None:
        logger.debug(f'mark notification as read: {source_type}, {notification_ids}')
        async with self._http_client.post(
            f'{self._config.url}/v1/mark_notification_as_read_internal',
            json={
                'source_type': source_type,
                'notification_ids': notification_ids,
            },
        ) as response:
            logger.debug(f'mark notification as read done, status: {response.status}')
            response.raise_for_status()
            return await response.json()
