#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mike Orlov <m.orlov@abm-jsc.ru>
#
import datetime
import json
from dataclasses import is_dataclass, Field
from decimal import Decimal
from functools import partial
from typing import Union

from .dict_to_dataclass import get_dataclass_field_name_to_field

Jsonable = Union[list, dict, str, int, float, None]


def read_json_file_by_path(path: str) -> Jsonable:
    with open(path) as file:
        return json.load(file)


class ReprInDumps:
    def __repr_in_dumps__(self):
        return repr(self)


class EnhancedJSONEncoder(json.JSONEncoder):
    def default(self, o):
        if isinstance(o, ReprInDumps):
            return o.__repr_in_dumps__()
        if isinstance(o, type):
            return o.__name__
        if is_dataclass(o):
            # d = asdict(o)
            d = {}
            dataclass_fields = get_dataclass_field_name_to_field(type(o), with_init_vars=False)
            for key in dataclass_fields:
                descriptor: Field = dataclass_fields[key]
                if descriptor.repr:
                    d[key] = getattr(o, key)
            return d
        if isinstance(o, datetime.datetime):
            return int(o.timestamp() * 1000)
        if isinstance(o, datetime.date):
            return o.isoformat()
        if isinstance(o, datetime.time):
            return o.isoformat()
        if isinstance(o, set):
            return list(o)
        if isinstance(o, Decimal):
            return str(o)
        return super().default(o)


custom_dumps = partial(json.dumps, cls=EnhancedJSONEncoder)
