#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>

import logging
from dataclasses import dataclass
from io import BytesIO
from typing import List, Union, Optional

from dict_caster.extras import to_list
from init_helpers.dict_to_dataclass import dict_to_dataclass

from ..exceptions import FileAlreadyExists, PathDoesNotExists
from ..s3_file_storage.s3_connector import S3Connector
from ..exceptions import BucketAlreadyExist

logger = logging.getLogger(__name__)


@dataclass
class SuccessfullyDeletedFile:
    Key: str


@dataclass
class UnsuccessfullyDeletedFile:
    Key: str
    Code: str


@dataclass
class DeleteAnswer:
    Deleted: list[SuccessfullyDeletedFile]
    Errors: list[UnsuccessfullyDeletedFile]


class S3Interface:
    @dataclass
    class Config(S3Connector.Config):
        default_acl: str = "private"

    def __init__(self, config):
        logger.info(f"{type(self).__name__} init")
        self._s3_connector = S3Connector(config)
        self._config = config

    async def upload_file(self, bucket_name: str, payload: bytes, key: str, allow_rewrite: bool = False) -> None:
        logger.info(f"Uploading file: bucket_name = {bucket_name}, key = {key}")
        async with self._s3_connector.session as session:
            bucket = await session.Bucket(bucket_name)

            if any([obj.key == key async for obj in bucket.objects.filter(Prefix=key)]) and not allow_rewrite:
                raise FileAlreadyExists(key)

            await bucket.put_object(Body=payload, Key=key)

    async def get_file(self, bucket_name: str, key: str) -> bytes:
        logger.info(f"Getting file: bucket_name = {bucket_name}, key = {key}")
        async with self._s3_connector.session as session:
            bucket = await session.Bucket(bucket_name)

            if not any([obj.key == key async for obj in bucket.objects.filter(Prefix=key)]):
                raise PathDoesNotExists(key)

            payload = BytesIO()
            await bucket.download_fileobj(key, payload)
            return payload.getvalue()

    async def delete_files(self, bucket_name: str, keys: Union[List[str], str]) -> DeleteAnswer:
        logger.info(f"Deleting files: bucket_name = {bucket_name}, keys = {keys}")
        keys = to_list(keys)
        async with self._s3_connector.session as session:
            bucket = await session.Bucket(bucket_name)
            del_object = {"Objects": [{"Key": key} for key in keys]}
            delete_response = await bucket.delete_objects(Delete=del_object)
        return dict_to_dataclass(delete_response, DeleteAnswer)

    async def check_file_existence(self, bucket_name: str, key: str) -> bool:
        logger.info(f"Checking file existence: bucket_name = {bucket_name}, key = {key}")
        async with self._s3_connector.session as session:
            bucket = await session.Bucket(bucket_name)

            is_file_existence = False
            if any([obj.key == key async for obj in bucket.objects.filter(Prefix=key)]):
                is_file_existence = True

            return is_file_existence

    async def create_bucket(self, bucket_name: str, acl: Optional[str] = None) -> None:
        """
        :param bucket_name: str
        :param acl: str 'private'|'public-read'|'public-read-write'|'authenticated-read'
        :return:
        """
        if not acl:
            acl = self._config.default_acl
        logger.info(f"Creating bucket: bucket_name = {bucket_name}, acl = {acl}")
        async with self._s3_connector.session as session:
            async for bucket in session.buckets.all():
                if bucket.name == bucket_name:
                    raise BucketAlreadyExist(bucket_name)
            bucket = await session.Bucket(bucket_name)
            await bucket.create(ACL=acl,
                                CreateBucketConfiguration={'LocationConstraint': self._s3_connector.region_name})
