#  Copyright (C) 2022
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasiliev Ivan <i.vasiliev@technokert.ru>
import json
from abc import ABC, abstractmethod
from dataclasses import dataclass
from typing import Optional, Any, Union

from aiokafka import ConsumerRecord

from .record_processor.kafka_record import PreproceededRecord


class AbstractDeserializer(ABC):

    @abstractmethod
    def deserialize(self,  payload: bytes) -> Any:
        pass


class StringDeserializer(AbstractDeserializer):
    def __init__(self, charset: str = "utf-8") -> None:
        self.charset = charset

    def deserialize(self, payload: bytes) -> str:
        answer = payload.decode(self.charset)
        return answer


class JsonDeserializer(StringDeserializer):
    def deserialize(self, payload: bytes) -> Union[dict, str, int]:
        value = super().deserialize(payload)
        if value is not None:
            return json.loads(value)


class RecordDeserializer:
    @dataclass
    class Context:
        key_deserializer: Optional[AbstractDeserializer] = None
        value_deserializer: Optional[AbstractDeserializer] = None

    def __init__(self, context: Context) -> None:
        self.context = context

    def deserialize_record(self, record: ConsumerRecord) -> PreproceededRecord:
        return PreproceededRecord(key=self._deserialize_key(record.key), value=self._deserialize_value(record.value))

    def _deserialize_key(self, key: Optional[bytes]) -> Optional[Any]:
        deserializer = self.context.key_deserializer
        return key if key is None or deserializer is None else deserializer.deserialize(key)

    def _deserialize_value(self, value: Optional[bytes]) -> Optional[Any]:
        deserializer = self.context.value_deserializer
        return value if value is None or deserializer is None else deserializer.deserialize(value)
