#  Copyright (C) 2023
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>

from dataclasses import dataclass
from enum import unique, StrEnum
from typing import Union, Protocol, Any, TypeVar, Type

from init_helpers.dict_to_dataclass import convert_to_type


@unique
class ComparisonOperator(StrEnum):
    eq = '='
    equiv = '=='
    ne = '!='
    lt = '<'
    le = '<='
    ge = '>='
    gt = '>'
    in_ = 'in'
    is_null = 'is_null'
    ilike = 'ilike'


class EntityConditionProtocol(Protocol):
    attribute: str
    value: Union[int, float, bool, str, None, list[Union[int, float, bool, str]]]
    operator: ComparisonOperator


@dataclass
class EntityOrderProtocol(Protocol):
    column: str
    ascending: bool


T = TypeVar('T')


def entity_element_to_dict(entity_element: T, entity_element_protocol_type: type) -> dict[str, Any]:
    result = {}
    for field_name, field_type in entity_element_protocol_type.__annotations__.items():
        result[field_name] = convert_to_type(field_type, field_name, getattr(entity_element, field_name))
    return result


def entity_condition_as_dict(entity_condition: EntityConditionProtocol) -> dict[str, Any]:
    return entity_element_to_dict(entity_condition, EntityConditionProtocol)


def entity_order_as_dict(entity_order: EntityOrderProtocol) -> dict[str, Any]:
    return entity_element_to_dict(entity_order, EntityOrderProtocol)


@dataclass
class EntityCondition(EntityConditionProtocol):
    attribute: str
    value: Union[int, float, bool, str, None, list[Union[int, float, bool, str]]]
    operator: ComparisonOperator = ComparisonOperator.eq


@dataclass
class EntityOrder(EntityOrderProtocol):
    column: str
    ascending: bool = True
