#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>

import logging
from dataclasses import dataclass
from typing import List, Union, Optional, Dict

from botocore.client import BaseClient
from botocore.exceptions import ClientError

from ..exceptions import BucketAlreadyExist
from ..exceptions import FileAlreadyExists, PathDoesNotExists
from ..s3_file_storage.s3_connector import S3Connector

logger = logging.getLogger(__name__)


@dataclass
class Range:
    offset: int
    size: int

    def get_range_header(self) -> Dict[str, str]:
        byte_range = f'bytes={self.offset}-'
        byte_range = f'{byte_range}{self.size}' if self.size != -1 else byte_range
        return {'Range': byte_range}


class S3Interface:
    @dataclass
    class Config(S3Connector.Config):
        default_acl: str = "private"

    def __init__(self, config):
        logger.info(f"{type(self).__name__} init")
        self._s3_connector = S3Connector(config)
        self._default_acl = config.default_acl

    async def upload_file(self, bucket_name: str, payload: bytes, key: str, allow_rewrite: bool = False) -> None:
        logger.info(f"Uploading file: bucket_name = {bucket_name}, key = {key}")
        async with self._s3_connector.client as s3_client:
            if await self._check_file_exists(bucket_name, key, s3_client) and not allow_rewrite:
                raise FileAlreadyExists(key)

            await s3_client.put_object(Body=payload, Bucket=bucket_name, Key=key)

    async def get_file(self, bucket_name: str, key: str, range_: Optional[Range] = None) -> bytes:
        logger.info(f"Getting file: bucket_name = {bucket_name}, key = {key}")
        headers = {}
        if range_ is not None:
            headers.update(range_.get_range_header())

        async with self._s3_connector.client as s3_client:
            if not await self._check_file_exists(bucket_name, key, s3_client):
                raise PathDoesNotExists(key)

            response = await s3_client.get_object(Bucket=bucket_name, Key=key)
            body = await response['Body'].read()
            return body

    async def delete_file(self, bucket_name: str, keys: Union[List[str], str]) -> None:
        logger.info(f"Deleting files: bucket_name = {bucket_name}, keys = {keys}")
        if not isinstance(keys, list):
            keys = [keys]
        async with self._s3_connector.client as s3_client:
            del_object = {"Objects": [{"Key": key} for key in keys]}
            # TODO mb add file exists check
            await s3_client.delete_objects(Bucket=bucket_name, Delete=del_object)

    # async def check_file_existence(self, bucket_name: str, key: str) -> bool:
    #     logger.info(f"Checking file existence: bucket_name = {bucket_name}, key = {key}")
    #     async with self._s3_connector.client as session:
    #         bucket = await session.Bucket(bucket_name)
    #
    #         is_file_existence = False
    #         if any([obj.key == key async for obj in bucket.objectss.filter(Prefix=key)]):
    #             is_file_existence = True
    #
    #         return is_file_existence

    async def create_bucket(self, bucket_name: str, acl: Optional[str] = None) -> None:
        """
        :param bucket_name: str
        :param acl: str 'private'|'public-read'|'public-read-write'|'authenticated-read'
        :return:
        """
        if not acl:
            acl = self._default_acl
        logger.info(f"Creating bucket: bucket_name = {bucket_name}, acl = {acl}")
        async with self._s3_connector.client as s3_client:
            try:
                if await self._check_bucket_exists(bucket_name, s3_client):
                    raise BucketAlreadyExist(bucket_name)
            except ClientError as exc:
                logger.error(f"ClientError: {exc.response['Error']['Code']} - {exc.response['Error']['Message']}")
                raise exc

            await s3_client.create_bucket(
                ACL=acl,
                Bucket=bucket_name,
                CreateBucketConfiguration={'LocationConstraint': self._s3_connector.region_name},
            )

    async def _check_file_exists(self, bucket_name: str, key: str, s3_client: Optional[BaseClient] = None) -> bool:
        logger.info(f"Checking file existence: bucket_name = {bucket_name}, key = {key}")
        try:
            if s3_client is not None:
                await s3_client.head_object(Bucket=bucket_name, Key=key)
            else:
                async with self._s3_connector.client as s3_client:
                    await s3_client.head_object(Bucket=bucket_name, Key=key)
            return True
        except ClientError as exc:
            if exc.response['Error']['Code'] == '404':
                logger.info(f"File '{key}' does not exist in bucket '{bucket_name}'.")
                return False
            logger.error(
                f"Error checking file existence: {exc.response['Error']['Code']} - {exc.response['Error']['Message']}"
            )
            raise exc

    @staticmethod
    async def _check_bucket_exists(bucket_name: str, s3_client: BaseClient) -> bool:
        try:
            await s3_client.head_bucket(Bucket=bucket_name)
            return True
        except ClientError as exc:
            if exc.response['Error']['Code'] == '404':
                return False
            raise exc
