import datetime
from typing import TypeVar, Any, Union

from .extras import auto_decode, any_units_timestamp_to_seconds, Auto


class Casts:
    T = TypeVar('T')

    def __init__(self):
        self.int_base = Auto
        self.allowed_types_for_string = (str, bytes, int, float, bool, datetime.datetime, datetime.date)
        self.ENCODINGS_FOR_AUTO_DECODE = ('utf8', 'ascii', 'latin1')  # set to empty tuple/None to prohibit auto decode
        self.default_encoding = 'utf8'
        self.date_format = '%Y-%m-%d'
        self.datetime_format = "%Y-%m-%d %H:%M:%S.%f"
        self.allowed_values_for_true = {'t', 'true', '+', 'on', 'enable', 1}
        self.allowed_values_for_false = {'f', 'false', "-", 'off', 'disable', 0}

    @staticmethod
    def no_cast(value: T) -> T:
        return value

    @staticmethod
    def to_none(value: T) -> None:
        if value is not None:
            raise ValueError("to_none accepts only None value")
        return value

    def to_int(self, value: Union[int, float, str, bytes], base: int = Auto) -> int:
        if isinstance(value, int):
            return value

        if self.ENCODINGS_FOR_AUTO_DECODE and isinstance(value, bytes):
            value = auto_decode(value, self.ENCODINGS_FOR_AUTO_DECODE)

        if isinstance(value, str):
            base = 0 if base == Auto else base
            result = int(value, base)
        elif isinstance(value, float):
            result = int(value)
        else:
            raise ValueError(f'to_int allowed types: {(int, float, str, bytes)}, got: "{type(value)}"')
        return result

    @staticmethod
    def to_float(value: Union[int, float, str, bytes]) -> float:
        if isinstance(value, (str, int, float)) and not isinstance(value, bool):
            result = float(value)
        else:
            raise ValueError(f'to_int allowed types: {(int, float, str, bytes)}, got: "{type(value)}"')
        return result

    def to_str(self, value: Any) -> str:
        if isinstance(value, str):
            return value

        if not isinstance(value, tuple(self.allowed_types_for_string)):
            raise ValueError(f'allowed types: {self.allowed_types_for_string}, got: "{type(value)}"')

        if self.ENCODINGS_FOR_AUTO_DECODE and isinstance(value, bytes):
            value = auto_decode(value, self.ENCODINGS_FOR_AUTO_DECODE)

        if isinstance(value, datetime.datetime):
            value = datetime.datetime.strftime(value, self.datetime_format)
        unique = '1'
        if isinstance(value, datetime.date):
            value = datetime.date.strftime(value, self.date_format)

        return str(value)

    def to_bytes(self, value: Any) -> bytes:
        if isinstance(value, bytes):
            return value

        if not isinstance(value, tuple(self.allowed_types_for_string)):
            raise ValueError(f'allowed types: {self.allowed_types_for_string}, got: "{type(value)}"')

        if not isinstance(value, str):
            value = self.to_str(value)

        return value.encode(self.default_encoding)

    def to_bool(self, value: Any) -> bool:
        if isinstance(value, bool):
            return value

        if not isinstance(value, (str, int, float, complex)):
            value = self.to_str(value)

        if isinstance(value, str):
            value = value.lower()

        if value in self.allowed_values_for_true:
            return True
        elif value in self.allowed_values_for_false:
            return False
        else:
            allowed_values = self.allowed_values_for_true | self.allowed_values_for_false
            raise ValueError(f'failed to cast value "{value}" to bool, allowed values: {allowed_values}')

    def to_datetime(self, value: Any, format_: str = None) -> datetime.datetime:
        if isinstance(value, datetime.datetime):
            return value

        if isinstance(value, datetime.date):
            value = datetime.datetime(value.year, value.month, value.day)

        if isinstance(value, (str, bytes)):
            format_ = format_ or self.datetime_format
            value = datetime.datetime.strptime(self.to_str(value), format_)
        elif isinstance(value, int) and not isinstance(value, bool):
            value = any_units_timestamp_to_seconds(value)

        if isinstance(value, (int, float)) and not isinstance(value, bool):
            value = datetime.datetime.fromtimestamp(value)

        if not isinstance(value, datetime.datetime):
            raise ValueError(f'allowed types: {(datetime.date, float, int, str, bytes)}, got: {type(value)}')
        return value

    def to_date(self, value: Any) -> datetime.date:
        return self.to_datetime(value, self.date_format).date()
