#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mike Orlov <m.orlov@abm-jsc.ru>
#
import json
import logging
from dataclasses import dataclass
from typing import Mapping, Optional, Any

import furl as furl
from aiohttp import ClientSession, BasicAuth
from async_tools import AsyncOnStop
from init_helpers.custom_json import custom_dumps, Jsonable
from multidict import MultiMapping

from http_tools import HttpClient

logger = logging.getLogger(__name__)

DEFAULT_PROTOCOL = "http"
PROTOCOL_SEPARATOR = "://"


class HttpServerConnector(AsyncOnStop):
    @dataclass
    class Config:
        location: str = ""  # deprecated, use url instead
        url: str = ""  # use it
        timeout_sec: float = 100
        max_connections: int = None

        def __post_init__(self):
            dirty_url = self.url or self.location
            if dirty_url is None:
                raise ValueError('required "url" or "location", got None')
            parts = dirty_url.split(PROTOCOL_SEPARATOR, 1)
            protocol = parts[0] if len(parts) > 1 else DEFAULT_PROTOCOL
            non_protocol = parts[-1]
            self.url = f'{protocol}{PROTOCOL_SEPARATOR}{non_protocol}'
            self.location = self.url  # compatibility

    @dataclass
    class Context:
        session: ClientSession

    def __init__(self, config: Config, context: Context):
        logger.info(f"{type(self).__name__} init")
        self.config = config
        self._requester = HttpClient(context.session, config.timeout_sec, config.max_connections)

    async def _on_stop(self):
        logger.info(f"{type(self).__name__} stop")
        await self._requester._on_stop()

    def _get_url(self, path: str) -> str:
        location_furl = furl.furl(self.config.url) / path
        location_furl.path.normalize()
        return location_furl.url

    async def post_json(self, path: str, payload: Jsonable, url_query: MultiMapping = None,
                        headers: Optional[Mapping[str, str]] = None, auth: Optional[BasicAuth] = None) -> Any:
        jsom_payload = custom_dumps(payload)
        payload = json.loads(jsom_payload)
        url = self._get_url(path)
        result = await self._requester.post(url, params=url_query, json=payload, headers=headers, auth=auth)
        return result

    async def put_json(self, path: str, payload: Jsonable, url_query: MultiMapping = None,
                       headers: Optional[Mapping[str, str]] = None, auth: Optional[BasicAuth] = None) -> Any:
        jsom_payload = custom_dumps(payload)
        payload = json.loads(jsom_payload)
        url = self._get_url(path)
        result = await self._requester.put(url, params=url_query, json=payload, headers=headers, auth=auth)
        return result

    async def post(self, path: str, payload: bytes, url_query: Mapping = None,
                   headers: Optional[Mapping[str, str]] = None, auth: Optional[BasicAuth] = None) -> Any:
        url = self._get_url(path)
        result = await self._requester.post(url, params=url_query, data=payload, headers=headers, auth=auth)
        return result

    async def delete(self, path: str, payload: bytes = None, url_query: Mapping = None,
                     headers: Optional[Mapping[str, str]] = None, auth: Optional[BasicAuth] = None) -> Any:
        url = self._get_url(path)
        result = await self._requester.delete(url, params=url_query, data=payload, headers=headers, auth=auth)
        return result

    async def get(self, path: str, url_query: Mapping = None, headers: Optional[Mapping[str, str]] = None,
                  auth: Optional[BasicAuth] = None) -> Any:
        url = self._get_url(path)
        result = await self._requester.get(url, params=url_query, headers=headers, auth=auth)
        return result

    async def patch_json(self, path: str, payload: Jsonable, url_query: MultiMapping = None,
                         headers: Optional[Mapping[str, str]] = None, auth: Optional[BasicAuth] = None) -> Any:
        jsom_payload = custom_dumps(payload)
        payload = json.loads(jsom_payload)
        url = self._get_url(path)
        result = await self._requester.patch(url, params=url_query, json=payload, headers=headers, auth=auth)
        return result
