#  Copyright (C) 2024
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mikhail Mamrov <m.mamrov@abm-jsc.ru>
#
import logging
from dataclasses import dataclass
from enum import Enum, unique
from typing import TypeAlias

from aiohttp import hdrs
from http_tools import AbmServiceConnector
from init_helpers.dict_to_dataclass import dict_to_dataclass
from yarl import URL

logger = logging.getLogger(__name__)


def construct_origin_from_referer(referer: str) -> str:
    return str(URL(referer).origin())


TokenId: TypeAlias = str


@unique
class TokenType(str, Enum):
    ACCESS_TOKEN = "access_token"
    REFRESH_TOKEN = "refresh_token"


@dataclass(frozen=True)
class JWTPortalPayload:
    id: int
    origin: str


@dataclass(frozen=True)
class JWTPortalMemberPayload:
    id: int
    role_id: int
    registered_at: int


@dataclass(frozen=True)
class JWTPayload:
    portal: JWTPortalPayload
    portal_member: JWTPortalMemberPayload
    id: TokenId
    type: TokenType
    iat: int
    exp: int


class AuthServerConnector:
    Config = AbmServiceConnector.Config
    Context = AbmServiceConnector.Context

    def __init__(self, config: Config, context: Context) -> None:
        self.config = config
        self.context = context
        self._connector = AbmServiceConnector(config, context)
        logger.info(f"{type(self).__name__} inited")

    async def validate_access_token(self, token: str, origin: str) -> JWTPayload:
        answer = await self._connector.post(
            path="v1/access_token/validate",
            payload=None,
            headers={
                hdrs.ORIGIN: origin,
                hdrs.AUTHORIZATION: f"Bearer {token}",
            },
        )
        return dict_to_dataclass(answer, JWTPayload)
