#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors:
#  Vasya Svintsov <v.svintsov@techokert.ru>,
#  Alexander Medvedev <a.medvedev@abm-jsc.ru>,
#  Andrey Vaydich <a.vaydich@abm-jsc.ru>
import re
import traceback
from dataclasses import dataclass
from typing import Optional

import yarl
from aiohttp.hdrs import RANGE, CONTENT_LENGTH, IF_NONE_MATCH
from aiohttp.web_exceptions import HTTPUnprocessableEntity
from dict_caster import Item, DictCaster
from http_tools import HttpServer, Answer, FileAnswer, WrappedAnswer, HttpStatusCode
from http_tools.request import IncomingRequest

from .extra import assert_raise
from .entities.link_info import LinkInfo
from .controller import Controller
from .tools.file_extension import define_file_mime_type_by_extension
from .tools.error_middleware import error_middleware


class HttpHandler:
    @dataclass
    class Context:
        http_server: HttpServer
        controller: Controller

    def __init__(self, context: Context, path_prefix: Optional[str] = None) -> None:
        self.context = context

        path_prefix = path_prefix or '/'
        self.context.http_server.register_handler(str(yarl.URL(path_prefix) / 'file/add'), self.add)
        self.context.http_server.register_handler(str(yarl.URL(path_prefix) / 'file/get'), self.get, {"GET", "POST"})
        self.context.http_server.register_handler(str(yarl.URL(path_prefix) / 'file/get'), self.head, {"HEAD"})
        self.context.http_server.register_handler(str(yarl.URL(path_prefix) / 'file/preview'), self.get_thumbnail)
        self.context.http_server.register_handler(str(yarl.URL(path_prefix) / 'file/zip'), self.zip_files)
        self.context.http_server.register_handler(str(yarl.URL(path_prefix) / 'file/info'), self.info)
        self.context.http_server.register_handler(str(yarl.URL(path_prefix) / 'file/delete'), self.delete)

    @error_middleware
    async def add(self, request: IncomingRequest) -> Answer:
        if file := request.key_value_arguments.get('file'):  # multipart/form-data
            filename = file.filename
            content = file.content
        elif request.payload:  # application/octet-stream
            filename = DictCaster(Item('filename', str)).cast_and_return(request.key_value_arguments)
            content = request.payload
        else:
            raise HTTPUnprocessableEntity(reason='FileMissing')

        is_thumbnail_required = DictCaster(
            Item('is_thumbnail_required', bool, default=False),
        ).cast_and_return(request.key_value_arguments)

        try:
            link_info, thumbnail_info = await self.context.controller.upload(
                content=content, filename=filename, is_thumbnail_required=is_thumbnail_required)
        except Exception:
            traceback.print_exc()
            raise
        return WrappedAnswer({"original": link_info, "thumbnail": thumbnail_info})

    @error_middleware
    async def get(self, request: IncomingRequest) -> Answer:
        link_id = DictCaster(Item('id', str)).cast_and_return(request.key_value_arguments)
        byte_range = _parse_range_header(request.metadata.get_header(RANGE))
        possible_etags = _parse_if_none_match_header(request.metadata.get_header(IF_NONE_MATCH))
        link_info = await self.context.controller.head(link_id)
        assert_raise(link_info, FileNotFoundError, link_id)
        if link_info.file_info_key in possible_etags:
            return Answer(None, HttpStatusCode.NotModified, headers=generate_cache_headers(link_info))
        payload = await self.context.controller.read(link_info.file_info_key, byte_range)
        return FileAnswer(payload=payload, file_name=link_info.filename,
                          content_type=define_file_mime_type_by_extension(link_info.extension),
                          headers=generate_cache_headers(link_info))

    @error_middleware
    async def head(self, request: IncomingRequest) -> Answer:
        link_id = DictCaster(Item('id', str)).cast_and_return(request.key_value_arguments)
        link_info = await self.context.controller.head(link_id)
        return FileAnswer(payload=None, file_name=link_info.filename,
                          content_type=define_file_mime_type_by_extension(link_info.extension),
                          headers={CONTENT_LENGTH: str(link_info.file_info.size)})

    @error_middleware
    async def zip_files(self, request: IncomingRequest) -> Answer:
        ids = DictCaster(Item('ids', list[str])).cast_and_return(request.key_value_arguments)
        link_info, payload = await self.context.controller.zip_files(ids)
        return FileAnswer(payload=payload, file_name=link_info.filename,
                          content_type=define_file_mime_type_by_extension(link_info.extension))

    @error_middleware
    async def get_thumbnail(self, request: IncomingRequest) -> Answer:
        link_id = DictCaster(Item('id', str)).cast_and_return(request.key_value_arguments)
        link_info, payload = await self.context.controller.get_thumbnail(link_id)
        return FileAnswer(payload=payload, file_name=link_info.filename,
                          content_type=define_file_mime_type_by_extension(link_info.extension),
                          headers=generate_cache_headers(link_info))

    @error_middleware
    async def info(self, request: IncomingRequest) -> Answer:
        link_id = DictCaster(Item('id', str)).cast_and_return(request.key_value_arguments)
        link_info = await self.context.controller.head(link_id)
        return WrappedAnswer(link_info)

    @error_middleware
    async def delete(self, request: IncomingRequest) -> Answer:
        link_id = DictCaster(Item('id', str)).cast_and_return(request.key_value_arguments)
        link_info = await self.context.controller.delete(link_id)
        return WrappedAnswer(link_info)


def generate_cache_headers(link_info: LinkInfo) -> dict[str, str]:
    return {"Cache-Control": "max-age=10", "ETag": f'"{link_info.file_info_key}"'}


_range_header_regex = re.compile(r"^bytes=(\d+)-(\d+)?")
_if_none_match_header_regex = re.compile(r'"(.*?[^\\])"')


def _parse_range_header(header_value: str | None) -> range:
    if header_value and (match := _range_header_regex.match(header_value)):
        start, stop = match.groups()
        return range(int(start), -1 if stop is None else int(stop))


def _parse_if_none_match_header(header_value: str | None) -> list[str]:
    return list(_if_none_match_header_regex.findall(header_value)) if header_value else []
