#  Copyright (C) 2023
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mike Orlov <m.orlov@abm-jsc.ru>
from dataclasses import dataclass

from entity_read import sql
from entity_read.entity import Entity
from entity_read.sql.atoms import Selectable
from init_helpers import Jsonable

from .expression import Expression


@dataclass(frozen=True, repr=False)
class Literal(Expression):
    body: dict[str, Expression] | list[Expression] | Expression | str | float | int | bool | None

    def __repr_in_dumps__(self) -> Jsonable:
        return self._shortcut(self.body)

    def eval(self, entity_type: type[Entity], variables: dict[str, Selectable]) -> sql.atoms.literal.Literal:
        body = self.body
        if body is None:
            type_ = None
        elif isinstance(body, bool):
            type_ = bool
        elif isinstance(body, int):
            type_ = int
        elif isinstance(body, float):
            type_ = float
        elif isinstance(body, str):
            type_ = str
        elif isinstance(body, list):
            type_ = list
            # body = [v.eval(entity_type, variables).value for v in body] # on this line the code crashes with an error in new_immovable
            if all(isinstance(v, Expression) for v in body):
                body = [v.eval(entity_type, variables).value for v in body]
        else:
            raise TypeError(f"Unsupported literal type: {type(body)}")
        return sql.atoms.literal.Literal(type_=type_, value=body)

    @classmethod
    def _shortcut(cls, body: dict[str, Expression] | list[Expression] | Expression | str | float | int | bool | None) -> Jsonable:
        if body is None:
            return 'null'
        elif isinstance(body, bool):
            return 'true' if body else 'false'
        elif isinstance(body, int | float):
            return str(body)
        elif isinstance(body, str):
            return f"'{body}'"
        elif isinstance(body, list):
            return f"[{','.join([cls._shortcut(val) for val in body])}]"
        elif isinstance(body, dict):
            return {key: cls._shortcut(val) for key, val in body.items()}
        elif isinstance(body, Expression):
            return Expression.__repr_in_dumps__(body)
        else:
            raise TypeError(f"Unsupported literal type: {type(body)}")


def literal(body: dict[str, Expression | str | float | int | bool | None] | list[Expression | str | float | int | bool | None] | Expression | str | float | int | bool | None) -> Literal:
    return Literal(body=body)
