## Quick Start

#### EXAMPLE 1: Basic usage
Допустим, у нас есть http-сервер, который занимается отправкой писем. Нам требуется знать сколько отправок писем было выполнено.
```python
import asyncio
import random
from http_tools import HttpServer
from prometheus_tools.prometheus_controller import PrometheusController


async def send_email() -> bool:
    return bool(random.randint(0, 1))  # возвращаем статус отправки письма


async def main():
    http_server = HttpServer(HttpServer.Config(port=9999), HttpServer.Context(instance_id='example'))
    metrics = PrometheusController(PrometheusController.Config(), PrometheusController.Context(http_server))
    await http_server.async_init() # запускаем http_server

    while True:  # эмулируем выполнение работы сервиса
        await send_email()  # отправляем письмо
        metrics.counter('emails').inc()  # инкрементируем счетчик писем
        await asyncio.sleep(1)


asyncio.run(main())
```
Получение собранных метрик: 
```curl
curl --location 'http://0.0.0.0:9999/metrics'
```

#### EXAMPLE 2: Basic usage with labels
В первом примере метрика отправки писем не учитывает статус отправки письма. Исправим это с помощью лейблов.

```python
import asyncio
import random
from http_tools import HttpServer
from prometheus_tools.prometheus_controller import PrometheusController


async def send_email() -> bool:
    return bool(random.randint(0, 1))  # возвращаем статус отправки письма


async def main():
    http_server = HttpServer(HttpServer.Config(port=9999), HttpServer.Context(instance_id='example'))
    metrics = PrometheusController(PrometheusController.Config(), PrometheusController.Context(http_server))
    await http_server.async_init()

    while True:  # эмулируем выполнение работы сервиса
        sent_status = await send_email()  # отправляем письмо
        metrics.counter('emails', {'sent_status': sent_status}).inc()  # инкрементируем счетчик писем
        await asyncio.sleep(1)


asyncio.run(main())
```

**ВАЖНО!**
> Лейбл должен быть ограничен в возможных значениях. Каждое новое значение лейбла – это новый временной ряд. 


#### EXAMPLE 3: Track without labels
```python
import asyncio
import random
from http_tools import HttpServer
from prometheus_tools.prometheus_controller import PrometheusController


async def sleeper() -> None:
    await asyncio.sleep(random.random() * 2)


async def main():
    http_server = HttpServer(HttpServer.Config(port=9999), HttpServer.Context(instance_id='example'))
    metrics = PrometheusController(PrometheusController.Config(), PrometheusController.Context(http_server))
    await http_server.async_init()

    while True:  # эмулируем выполнение работы сервиса
        with metrics.track('sleeper'): # тречим выполнение задачи
            await sleeper() # выполняем задачу
        await asyncio.sleep(1)


asyncio.run(main())
```

#### EXAMPLE 4: Track with labels
```python
import asyncio
import random
from http_tools import HttpServer
from prometheus_tools.prometheus_controller import PrometheusController


async def send_email() -> bool:
    value = random.random()
    await asyncio.sleep(value)
    if value > 0.8:
        raise ValueError('some_error')
    return bool(random.randint(0, 1))


async def main():
    http_server = HttpServer(HttpServer.Config(port=9999), HttpServer.Context(instance_id='example'))
    metrics = PrometheusController(PrometheusController.Config(), PrometheusController.Context(http_server))
    await http_server.async_init()

    while True:
        for email_type in ('announcement', 'newsletter'):
            labels = {'type': email_type}
            with metrics.track('emails', labels, except_labels={'status': None, 'error': None}):
                try:
                    sent_status = await send_email()
                    labels['status'] = sent_status
                except Exception as er:
                    labels['error'] = type(er).__name__
        await asyncio.sleep(1)


asyncio.run(main())
```

#### EXAMPLE 5: Initer
```python
from dataclasses import dataclass
import init_helpers
from aiohttp import ClientSession
from async_tools import AsyncInitable, AsyncDeinitable
from http_tools import HttpServer
from prometheus_tools.prometheus_controller import PrometheusController


@dataclass
class Initer:
    @dataclass
    class Config:
        http_server: HttpServer.Config
        metrics: PrometheusController.Config

    config: Config

    @dataclass
    class Context(AsyncInitable, AsyncDeinitable):
        instance_id: str = "Example"
        http_server: HttpServer = None
        metrics: PrometheusController = None
        session: ClientSession = None

        def __post_init__(self) -> None:
            AsyncInitable.__init__(self)
            AsyncDeinitable.__init__(self)

    context: Context

    def __init__(self) -> None:
        self.config = init_helpers.parse_args(config_file=init_helpers.Arg.ini_file_to_dataclass(self.Config))
        self.context = self.Context()

    async def __aenter__(self) -> None:
        self.context.http_server = HttpServer(self.config.http_server, self.context)
        self.context.metrics = PrometheusController(self.config.metrics, self.context)
        self.session = self.context.metrics.get_monitored_client_session()
        await self.context.async_init()

    async def __aexit__(self, exc_type, exc_val, exc_tb) -> None:
        if self.context.session is not None:
            await self.context.session.close()
        await self.context.async_deinit()
```