#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>

import asyncio
import logging
from concurrent.futures import ThreadPoolExecutor
from dataclasses import dataclass
from typing import Optional
import aiofiles
from aiofiles.os import makedirs, remove, rmdir, listdir
from ..utils.to_list import to_list

from ..abstract_file_storage import AbstractFileStorage, FileSource, FileDeleteResult, FileDeleteStatus, \
    EmptyFileSourcesList, FileExistence
from ..exceptions import FileAlreadyExists, PathDoesNotExists

logger = logging.getLogger(__name__)


class DiscFileStorage(AbstractFileStorage):
    @dataclass
    class Config:
        location: str
        max_deleting_workers: Optional[int] = None

    def __init__(self, config: Config):
        logger.info(f"{type(self).__name__} init. Location: {config.location}")
        self._config = config

    async def save(self, content: bytes, relative_path: str, name: str, allow_rewrite: bool = False) -> None:
        dir_path = f"{self._config.location}/{relative_path}"

        if not await aiofiles.os.path.exists(dir_path):
            logger.debug(f"Path = {dir_path} does not exists. Will be created")
            await aiofiles.os.makedirs(dir_path)

        full_path = f"{dir_path}/{name}"
        logger.debug(f"Full path = {full_path}")
        if await aiofiles.os.path.exists(full_path) and not allow_rewrite:
            raise FileAlreadyExists(full_path)

        logger.debug(f"Saving file along the path: {full_path}")
        async with aiofiles.open(full_path, 'wb') as file:
            await file.write(content)

    async def load(self, relative_path: str, name: str, offset: int = 0, size: int = -1) -> bytes:
        dir_path = f"{self._config.location}/{relative_path}"
        full_path = f"{dir_path}/{name}"
        if not await aiofiles.os.path.exists(full_path):
            raise PathDoesNotExists(full_path)

        logger.debug(f"Loading file along the path: {full_path}")
        async with aiofiles.open(full_path, 'rb') as file:
            await file.seek(offset)
            return await file.read(size)

    async def delete(self, file_sources: list[FileSource] | FileSource) -> list[FileDeleteResult]:
        file_sources = to_list(file_sources)
        if not file_sources:
            raise EmptyFileSourcesList

        dir_paths = set()
        file_full_paths = []
        for file_source in file_sources:
            dir_path = f"{self._config.location}/{file_source.relative_path}"
            full_path = f"{dir_path}/{file_source.name}"
            file_full_paths.append(full_path)

        executor = ThreadPoolExecutor(self._config.max_deleting_workers)
        delete_file__tasks = [self._delete_file(full_path, executor) for full_path in file_full_paths]
        delete_file__task_results = await asyncio.gather(*delete_file__tasks, return_exceptions=True)
        file_delete_results = []
        for file_source, delete_file__task_result in zip(file_sources, delete_file__task_results):
            if delete_file__task_result is None:
                dir_paths.add(f"{self._config.location}/{file_source.relative_path}")
                status = FileDeleteStatus.OK
            else:
                status = FileDeleteStatus.FAILED
            file_delete_results.append(
                FileDeleteResult(file_source=file_source, status=status, reason=repr(delete_file__task_result))
            )
        logger.debug(f"File delete results: {file_delete_results}")

        await self._delete_empty_directories(dir_paths)
        return file_delete_results

    @staticmethod
    async def _delete_file(full_path: str, executor: ThreadPoolExecutor) -> None:
        if not await aiofiles.os.path.exists(full_path):
            raise PathDoesNotExists(full_path)
        await remove(full_path, executor=executor)

    @staticmethod
    async def _delete_empty_directories(dir_paths: set[str]) -> None:
        empty_dir_paths = [dir_path for dir_path in dir_paths if not await listdir(dir_path)]
        if not empty_dir_paths:
            return

        delete_dir_tasks = [rmdir(dir_path) for dir_path in empty_dir_paths]
        delete_dir_task_results = await asyncio.gather(*delete_dir_tasks, return_exceptions=True)

        dir_path_to_exception = {}
        successfully_deleted_dir_paths = []
        for dir_path, delete_dir_task_result in zip(empty_dir_paths, delete_dir_task_results):
            if delete_dir_task_result is None:
                successfully_deleted_dir_paths.append(dir_path)
            else:
                dir_path_to_exception[dir_path] = delete_dir_task_result
        if successfully_deleted_dir_paths:
            logger.debug(f"Successfully deleted empty directories: {successfully_deleted_dir_paths}")
        if dir_path_to_exception:
            logger.warning(f"Failed deleting empty directories: {dir_path_to_exception}")

    async def check_files_existence(self, file_sources: list[FileSource] | FileSource) -> list[FileExistence]:
        file_sources = to_list(file_sources)
        check_file__tasks = [
            aiofiles.os.path.exists(f"{self._config.location}/{file_source.relative_path}/{file_source.name}")
            for file_source in file_sources
        ]
        check_file__task_results = await asyncio.gather(*check_file__tasks)
        return [
            FileExistence(file_source=file_source, is_exist=result)
            for file_source, result in zip(file_sources, check_file__task_results)
        ]
