import dataclasses
import datetime
import functools
import logging
import types
import typing
from dataclasses import dataclass, field
from decimal import Decimal
from enum import Enum
from enum import StrEnum
from types import UnionType
from typing import ClassVar
from typing import Optional, Type, Any

from apispec import APISpec
from http_tools import Answer
from init_helpers.dict_to_dataclass import NoValue

from .extras import DataclassProtocol

logger = logging.getLogger(__name__)


class ParameterLocation(StrEnum):
    query = 'query'
    path = 'path'
    header = 'header'
    cookie = 'cookie'
    body = 'body'  # INNER: not present in specification, used inside library


@dataclass(frozen=True)
class Parameter:
    name: str
    schema: Type | UnionType
    in_: ParameterLocation
    body_mime_type: ClassVar[str] = None
    default: Any = NoValue

    @property
    def required(self):
        return self.default is NoValue


@dataclass(frozen=True)
class QueryParameter(Parameter):
    in_: ParameterLocation = ParameterLocation.query


@dataclass(frozen=True)
class HeaderParameter(Parameter):
    in_: ParameterLocation = ParameterLocation.header


@dataclass(frozen=True)
class PathParameter(Parameter):
    in_: ParameterLocation = ParameterLocation.path


@dataclass
class Content:
    mime_type: str
    schema: Type


# @dataclass
# class Response:
#     content: Content


# T = TypeVar('T')
# 
# # @dataclass
# class JsonResponse(Generic[T]):
#     mime_type: str = "application/json"
#     def __init__(self, payload):
#         self.payload = payload

@dataclass
class RequestBody:
    content: Content


@dataclass
class SecuritySchemeType(StrEnum):
    http = "http"
    api_key = "apiKey"
    oauth2 = "oauth2"
    open_id_connect = "openIdConnect"


@dataclass(frozen=True)
class SecurityScheme:
    name: str
    type: str
    scheme: str


@dataclass(frozen=True)
class BearerSecurityScheme(SecurityScheme):
    type: str = 'http'
    scheme: str = 'bearer'


@dataclass
class Security:
    scheme: SecurityScheme
    scopes: list[str] = field(default_factory=list)


@dataclass
class Endpoint:
    path: str
    method: str
    operation_id: str
    securities: list[Security]
    parameters: list[Parameter]
    request_body: Optional[RequestBody]
    code_to_answer: dict[int, Type[Answer]]


@dataclass
class GetIdsFromXlsxResponseSchema:
    done: bool
    result: list[int]


class OpenApiWrapper:
    @dataclasses.dataclass
    class Config:
        pass

    @dataclasses.dataclass
    class Context:
        company: str
        project_group: str
        project_name: str

    def __init__(self, config: Config, context: Context):
        self.version = self._get_version()
        self.config = config
        self.context = context
        self.endpoints = []
        self.dataclass_type_to_name: dict[type[DataclassProtocol] | UnionType, str] = {}

    def register_endpoint(self, endpoint: Endpoint) -> None:
        self.endpoints.append(endpoint)

    def register_dataclass(self, dataclass_type: type[DataclassProtocol], name: str) -> None:
        self.dataclass_type_to_name[dataclass_type] = name

    def get_spec_dict(self) -> dict:
        spec = APISpec(
            title=f"{self.context.company}:{self.context.project_group}:{self.context.project_name}",
            version=self.version,
            openapi_version="3.1.0"
        )
        for dataclass_type in self.dataclass_type_to_name:
            self._add_schema_to_spec(spec, dataclass_type)
        for endpoint in self.endpoints:
            self._add_endpoint_to_spec(spec, endpoint)
        return spec.to_dict()

    @staticmethod
    def _get_version() -> str:
        now = datetime.datetime.now()
        return f"{now.year - 2021}.{now.month}.{now.day}{now.hour:02d}{now.minute:02d}"

    def _add_endpoint_to_spec(self, spec: APISpec, endpoint: Endpoint) -> None:
        operation_dict = {
            "summary": "",
            "operationId": endpoint.operation_id,
            "security": self._add_operation_securities(spec, endpoint.securities),
            "parameters": self._add_operation_parameters(spec, endpoint.parameters),
            "responses": self._add_operation_code_to_answer(spec, endpoint.code_to_answer),
        }
        if request_body_datum := self._add_operation_request_body(spec, endpoint.request_body):
            operation_dict["requestBody"] = request_body_datum

        spec.path(endpoint.path, operations={endpoint.method: operation_dict})

    def _add_operation_parameters(self, spec: APISpec, parameters: list[Parameter]) -> list[dict[str, Any]]:
        result: list[dict[str, Any]] = []
        for param in parameters:
            schema_name = self._add_schema_to_spec(spec, param.schema, default=param.default)
            result.append({"name": param.name, "in": param.in_, "required": param.required, "schema": schema_name})
        return result

    def _add_operation_securities(self, spec: APISpec, securities: list[Security]) -> list[dict[str, list[str]]]:
        return [{self._add_security_schema_to_spec(spec, security.scheme): security.scopes} for security in securities]

    def _add_operation_code_to_answer(self, spec: APISpec, code_to_answer: dict[int, Type[Answer]]) -> dict[str, str]:
        result: dict[str, str] = {}
        for code, answer in code_to_answer.items():
            response_name = self._add_answer_to_spec(spec, answer)
            result[str(int(code))] = response_name
        return result

    def _add_operation_request_body(self, spec: APISpec, request_body: Optional[RequestBody]) -> dict[str, Any] | None:
        result = None
        if request_body is not None:
            schema_name = self._add_schema_to_spec(spec, request_body.content.schema)
            result = {"required": True, "content": {request_body.content.mime_type: {"schema": schema_name}}}
        return result

    @functools.cache
    def _add_schema_to_spec(self, spec: APISpec, schema: type | UnionType, default: Any = NoValue) -> dict:
        logger.debug('_add_schema_to_spec: %s', schema)

        origin_type = typing.get_origin(schema)
        type_args = typing.get_args(schema)
        if origin_type in (typing.Union, types.UnionType):
            description = {
                "anyOf": [self._add_schema_to_spec(spec, arg) for arg in type_args],
            }
        elif origin_type is list:
            assert type_args and len(type_args) == 1, f'bad list element type: {type_args}'
            description = {
                "type": "array",
                "items": self._add_schema_to_spec(spec, type_args[0])
            }
        elif origin_type is dict:
            assert type_args and len(type_args) == 2, f'bad dict item types: {type_args}'
            description = {
                "type": "object",
                "additionalProperties": self._add_schema_to_spec(spec, type_args[1])
            }
        elif origin_type is not None:
            raise TypeError(f"Unknown origin_type: {schema}")

        elif dataclasses.is_dataclass(schema):
            description = {
                "type": "object",
                "properties": {}
            }
            # noinspection PyDataclass
            for field_ in dataclasses.fields(schema):
                if field_.repr:
                    description["properties"][field_.name] = self._add_schema_to_spec(spec, field_.type)

            name = self.dataclass_type_to_name.get(schema, f"schema_{schema.__name__}")
            logger.debug('schema: %s, %s, %s', name, schema, list(description["properties"]))
            spec.components.schema(component_id=name, component=description)
            return spec.components.get_ref("schema", name)
        elif schema is bool:  # MUST be before int
            description = {"type": "boolean"}
        elif issubclass(schema, int):
            description = {
                "type": "integer",
                # "format": "int32"
            }
        elif issubclass(schema, float):
            description = {
                "type": "number",
                # "format": "int32"
            }
        elif issubclass(schema, bytes):
            description = {
                "type": "string",
                "format": "binary"
            }
        elif issubclass(schema, str):
            description = {"type": "string"}
        elif issubclass(schema, Decimal):
            description = {"type": "string", 'pattern': r'^\d*\.?\d*$'}
        elif schema in (None, type(None)):
            return {"type": "null"}
        else:
            raise TypeError(f"Unknown type: {schema}")

        if isinstance(schema, type) and issubclass(schema, Enum):
            description['enum'] = [e.value for e in schema]

        if default is None or isinstance(default, (int, float, str, list, dict)):
            description['default'] = default
        # if default == NoValue:  # TODO: think about it, should we place "required" inside schema properties or not?
        #     description['required'] = True

        return description

    @functools.cache
    def _add_answer_to_spec(self, spec: APISpec, answer: Type[Answer]) -> str:
        content_type = answer.get_class_content_type()
        schema = answer.get_class_payload_type()
        name = answer.__name__
        # response_schema = response.__args__[0]
        # response_type = response.__origin__
        # name = f"response_{response_type.__name__}[{response_schema.__name__}]"
        schema_name_or_description = self._add_schema_to_spec(spec, schema)

        spec.components.response(
            component_id=name, component={
                "description": "",
                "content": {
                    content_type: {"schema": schema_name_or_description}
                }
            }
        )
        return name

    @functools.cache
    def _add_security_schema_to_spec(self, spec: APISpec, security_schema: SecurityScheme) -> str:
        spec.components.security_scheme(
            component_id=security_schema.name,
            component={
                "description": "",
                "type": security_schema.type,
                "scheme": security_schema.scheme,
            })
        return security_schema.name

    @staticmethod
    def _get_name_for_object(object_: Any) -> str:
        return f"component_{object_.__name__}"
