#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>

from abc import ABC, abstractmethod
from dataclasses import dataclass
from enum import unique, IntEnum


@dataclass
class FileSource:
    relative_path: str
    name: str


@unique
class FileDeleteStatus(IntEnum):
    FAILED = 0
    OK = 1


@dataclass
class FileDeleteResult:
    file_source: FileSource
    status: FileDeleteStatus
    reason: Exception | str | None = None


class AbstractFileStorage(ABC):
    @abstractmethod
    async def save(self, content: bytes, relative_path: str, name: str, allow_rewrite: bool = False) -> None:
        pass

    @abstractmethod
    async def load(self, relative_path: str, name: str, offset: int = 0, size: int = -1) -> bytes:
        pass

    @abstractmethod
    async def delete(self, file_sources: list[FileSource] | FileSource) -> list[FileDeleteResult]:
        pass

    @abstractmethod
    async def check_file_existence(self, relative_path: str, name: str) -> None:
        pass
