#  Copyright (C) 2023
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Revva Konstantin <k.revva@abm-jsc.ru>


from dataclasses import dataclass
from typing import Type, Any, Optional
import json

from http_tools.http_server_connector import HttpServerConnector
from init_helpers.dict_to_dataclass import dict_to_dataclass, T
from init_helpers.custom_json import custom_dumps
from time_tools import Timerange

DetectorSettingsType = Type[T]


class DetectorHttpConnector:
    @dataclass
    class Context(HttpServerConnector.Context):
        project_name: str

    Config = HttpServerConnector.Config

    def __init__(self,
                 config: Config,
                 context: Context,
                 detector_settings_type: DetectorSettingsType) -> None:
        self.config = config
        self.context = context
        self._connector = HttpServerConnector(config, context)
        self.detector_settings_type = detector_settings_type

    async def get_settings(self, rtsp_urls: list[str], token: Optional[str] = None) -> list[T]:
        payload = {"rtsp_urls": rtsp_urls}

        if token is not None:
            payload.update({"token": token})

        answer = await self._connector.post_json("/settings/get",
                                                 payload=payload,
                                                 headers=self._construct_headers())

        return [dict_to_dataclass(settings, self.detector_settings_type) for settings in answer["result"]]

    async def set_settings(self, detector_settings: T, token: Optional[str] = None) -> T:
        payload = {"token": token} if token is not None else {}

        payload.update(json.loads(custom_dumps(detector_settings)))

        answer = await self._connector.post_json("/settings/set",
                                                 payload=payload,
                                                 headers=self._construct_headers())

        return dict_to_dataclass(answer["result"], self.detector_settings_type)

    async def autotune_settings(self, rtsp_url: str, miss_decision_timerange: Timerange,
                                token: Optional[str] = None) -> T:
        payload = {"token": token} if token is not None else {}

        payload.update({"rtsp_url": rtsp_url,
                        "miss_decision_timerange": json.loads(custom_dumps(miss_decision_timerange))})

        answer = await self._connector.post_json("/settings/autotune",
                                                 payload=payload,
                                                 headers=self._construct_headers())

        return dict_to_dataclass(answer["result"], self.detector_settings_type)

    def _construct_headers(self) -> dict[str, str]:
        return {"server_name": self.context.project_name}
