#  Copyright (C) 2024
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mikhail Mamrov <m.mamrov@abm-jsc.ru>
#

from dataclasses import dataclass
from enum import StrEnum, unique

from aiohttp import hdrs
from http_tools import AbmServiceConnector
from init_helpers.dict_to_dataclass import dict_to_dataclass


class TokenId(str):
    pass


@unique
class TokenType(StrEnum):
    access_token = "access_token"
    refresh_token = "refresh_token"


@dataclass
class PortalPayload:
    id: int
    origin: str


@dataclass
class PortalMemberPayload:
    id: int
    role_id: int


@dataclass(frozen=True)
class JWTPayload:
    portal: PortalPayload
    portal_member: PortalMemberPayload
    id: TokenId
    type: TokenType
    iat: int
    exp: int


class AuthServerConnector:
    Config = AbmServiceConnector.Config
    Context = AbmServiceConnector.Context

    def __init__(self, config: Config, context: Context) -> None:
        self.config = config
        self.context = context
        self._connector = AbmServiceConnector(config, context)

    async def verify_token(self, token: str, origin: str) -> JWTPayload:
        answer = await self._connector.post(
            "v1/auth/verify",
            payload={"access_token": token},
            headers={hdrs.ORIGIN: origin},
        )
        print(f"{answer = }")
        return dict_to_dataclass(answer, JWTPayload)
