#  Copyright (C) 2023
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mike Orlov <m.orlov@abm-jsc.ru>
from collections.abc import Iterable
from dataclasses import dataclass, field

from entity_read import sql
from entity_read.entity import Entity
from entity_read.sql.atoms import Selectable

from ._eval_to_selector import eval_to_selector
from .alias import Alias
from .expression import Expression
from .order import Order
from .query_token import QueryToken
from .reference import Reference
from .relation import Relation


@dataclass(frozen=True, kw_only=True, repr=False)
class SubQuery(Expression):
    over: Relation | Reference
    attrs: list[Expression] | Expression = field(default_factory=list)
    vars: dict[str, Expression] = field(default_factory=dict)
    filters: list[Expression] = field(default_factory=list)
    searches: list[Expression] = field(default_factory=list)
    orders: list[Order] = field(default_factory=list)
    limit: int | None = None
    offset: int | None = None

    def __repr_in_dumps__(self) -> str | dict:
        over = self.over.__repr_in_dumps__()
        attrs = [a.__repr_in_dumps__() for a in self.attrs] if self.attrs else None
        filters = [f.__repr_in_dumps__() for f in self.filters] if self.filters else None
        searches = [s.__repr_in_dumps__() for s in self.searches] if self.searches else None
        orders = [o.shortcut() for o in self.orders] if self.orders else None  # TODO: check if shortcut is ok

        stringable = not self.vars
        stringable &= not attrs or all(isinstance(a, str) for a in attrs)
        stringable &= not filters or all(isinstance(f, str) for f in filters)
        stringable &= not searches or all(isinstance(s, str) for s in searches)
        stringable &= not orders or all(isinstance(o, str) for o in orders)

        if stringable:
            dirty_parts = [
                f"attrs=[{','.join(attrs)}]" if attrs else "",
                f"filters=[{','.join(filters)}]" if filters else "",
                f"searches=[{','.join(searches)}]" if searches else "",
                f"orders=[{','.join(orders)}]" if orders else "",
            ]
            return f"{over}.subquery({','.join([p for p in dirty_parts if p])})"
        return super().__repr_in_dumps__()

    def eval(self, entity_type: type[Entity], variables: dict[str, Selectable]) -> sql.lower.LowerSelector:
        result = eval_to_selector(
            entity_type=self.over.get_related_entity(entity_type),
            local_column_to_remote=self.over.get_remote_column_name_to_local(entity_type),  # relatively to child
            my_step=self.over.key, rename=self.shortcut(), is_scalar=isinstance(self.over, Reference),
            attrs=self.attrs, vars_=self.vars, filters=self.filters, searches=self.searches,
            orders=self.orders, limit=self.limit, offset=self.offset
        )
        return result

    def alias(self, key: str) -> Alias:
        return Alias(body=self, key=key)

    def shortcut(self) -> str:
        parts = []
        parts += prepare_tokens('attrs', self.attrs)
        parts += [f'vars={",".join([f"{k}={v.shortcut()}" for k, v in self.vars.items()])}'] if self.vars else [] # TODO CHECK ??? in report_export said assert not self.vars, "shortcut cannot be used with vars"
        parts += prepare_tokens('filters', self.filters)
        parts += prepare_tokens('searches', self.searches)
        parts += prepare_tokens('orders', self.orders)
        parts += [f'limit={self.limit}'] if self.limit else []
        parts += [f'offset={self.offset}'] if self.limit else []
        return f"{self.over.shortcut_over()}.subquery({','.join(parts)})"


def prepare_tokens(key: str, vals: list[QueryToken] | None) -> list[str]:
    if isinstance(vals, Iterable):
        return [f'{key}=[{",".join([val.shortcut() for val in vals])}]'] if vals else []
    elif isinstance(vals, QueryToken):
        return [f'{key}={vals.shortcut()}']
    else:
        return []
