#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>

import logging
from dataclasses import dataclass

from async_tools import AsyncInitable

from ..abstract_file_storage import AbstractFileStorage
from ..exceptions import BucketAlreadyExist
from ..s3_file_storage.s3_interface import S3Interface, Range

logger = logging.getLogger(__name__)


class S3FileStorage(AbstractFileStorage, AsyncInitable):
    @dataclass
    class Config(AbstractFileStorage.Config, S3Interface.Config):
        bucket_name: str = "abm-storage"

    Context = AbstractFileStorage.Context

    def __init__(self, config: Config, context: Context):
        AsyncInitable.__init__(self)
        AbstractFileStorage.__init__(self, config, context)
        logger.info(f"{type(self).__name__} init. Config: {config}")
        self._s3_interface = S3Interface(config)

    async def _async_init(self) -> None:
        try:
            await self._s3_interface.create_bucket(self.config.bucket_name)
        except BucketAlreadyExist:
            logger.info(f"Bucket already exist. bucket_name = {self.config.bucket_name}")
        else:
            logger.info(f"Bucket created. bucket_name = {self.config.bucket_name}")

    async def _save(self, key: str, value: bytes, allow_rewrite: bool = False) -> None:
        await self._s3_interface.upload_file(bucket_name=self.config.bucket_name,
                                             payload=value,
                                             key=self._get_storage_key(key),
                                             allow_rewrite=allow_rewrite,
                                             )

    async def _load(self, key: str, offset: int = 0, size: int = -1) -> bytes:
        return await self._s3_interface.get_file(bucket_name=self.config.bucket_name,
                                                 key=self._get_storage_key(key),
                                                 range_=Range(offset, size),
                                                 )

    async def _delete(self, key: str) -> None:
        return await self._s3_interface.delete_file(bucket_name=self.config.bucket_name,
                                                    keys=self._get_storage_key(key),
                                                    )

    async def _check_file_existence(self, key: str) -> bool:
        return await self._s3_interface.check_file_existence(bucket_name=self.config.bucket_name,
                                                             key=self._get_storage_key(key),
                                                             )

    def _get_storage_key(self, key: str) -> str:
        return '/'.join((self.config.root_dir, *self._split_key(key)))
