#  Copyright (C) 2024
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mike Orlov <m.orlov@abm-jsc.ru>

from sqlalchemy import String, Integer
from sqlalchemy_tools.entity_helpers.decorator import sqlalchemy_dataclass
from sqlalchemy_tools.entity_helpers.fields import sql_field, make_field__created_at
from sqlalchemy_tools.entity_helpers.sqlalchemy_base import sqlalchemy_mapper_registry

from ..tools.hashes import generate_md5_hash, generate_sha512_hash


@sqlalchemy_dataclass(sqlalchemy_mapper_registry)
class FileInfo:
    key: str = sql_field(String, primary_key=True, nullable=False, allow_init_primary_key=True)
    md5: str = sql_field(String, nullable=False)
    sha512: str = sql_field(String, nullable=False)
    size: int = sql_field(Integer, nullable=False)
    created_at: float = make_field__created_at()

    @classmethod
    def from_bytes(cls, value: bytes, **kwargs) -> 'FileInfo':
        size = len(value)
        md5_hash = generate_md5_hash(value)
        sha512_hash = generate_sha512_hash(value)
        return FileInfo(key=f'{md5_hash}_{size}', md5=md5_hash, sha512=sha512_hash, size=size, **kwargs)
