#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mike Orlov <m.orlov@abm-jsc.ru>
#
import asyncio
from logging import getLogger
from typing import Any

from .recursive import get_recursive

logger = getLogger(__name__)


class AsyncDeinitable:
    def __init__(self):
        self.async_deinit_scheduled__event: asyncio.Event = asyncio.Event()  # called async_deinit
        self._tasks_before_async_deinit: list[asyncio.Task] = []
        self.async_deinit_started__event: asyncio.Event = asyncio.Event()  # passed wait
        self.async_deinit_finished__event: asyncio.Event = asyncio.Event()  # done own _async_deinit

    def wait_before_async_deinit_for(self, task: asyncio.Task) -> None:
        self._tasks_before_async_deinit.append(task)

    async def async_deinit(self) -> None:
        if self.async_deinit_scheduled__event.is_set():
            logger.warning(f"async_start called extra time on {self}")
            return
        self.async_deinit_scheduled__event.set()
        if self._tasks_before_async_deinit:
            logger.debug(f"{self} before starting own _async_deinit is waiting for {self._tasks_before_async_deinit}")
            await asyncio.gather(*self._tasks_before_async_deinit)
        self.async_deinit_started__event.set()
        logger.debug(f"{self} started _async_deinit")
        await self._async_deinit()
        logger.debug(f"{self} finished _async_deinit")
        self.async_deinit_finished__event.set()

    async def _async_deinit(self) -> None:
        pass

    @staticmethod
    async def async_deinit_recursive(target: Any) -> None:
        deinitables = await get_recursive(target, condition=lambda x: isinstance(x, AsyncDeinitable))
        await asyncio.gather(*[x.async_deinit() for x in deinitables if not x.async_deinit_scheduled__event.is_set()])
