#  Copyright (C) 2025
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mike Orlov <m.orlov@abm-jsc.ru>
import abc

from ._token import Token, Getter, Call, Literal


class TokenFactory(abc.ABC):
    def make_getter(self, body: Token | None, key: str) -> Getter:
        raise NotImplementedError

    def make_call(self, body: Token, args: list[Token] | None = None, kwargs: dict[str, Token] | None = None) -> Call:
        raise NotImplementedError

    def make_literal(self, body: dict[str, Token] | list[Token] | str | float | int | bool | None) -> Literal:
        raise NotImplementedError


class DefaultTokenFactory(TokenFactory):
    def make_getter(self, body: Token | None, key: str) -> Getter:
        return Getter(body=body, key=key)

    def make_call(self, body: Token, args: list[Token] | None = None, kwargs: dict[str, Token] | None = None) -> Call:
        return Call(body=body, args=args or [], kwargs=kwargs or {})

    def make_literal(self, body: dict[str, Token] | list[Token] | str | float | int | bool | None) -> Literal:
        return Literal(body=body)


class OnlyPublicGetterTokenFactory(DefaultTokenFactory):
    """
    Prevents access to non-public attributes and values, like __module__ and __class__, otherwise usable for jailbreak
    """
    def make_getter(self, body: Token | None, key: str) -> Getter:
        if key.startswith("_"):
            raise KeyError(f"Attempt to access non public value: {key!r}")
        return super().make_getter(body, key)
