#  Copyright (C) 2023
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasiliev Ivan <i.vasiliev@technokert.ru>


import logging
from dataclasses import dataclass
from typing import Optional
from http_tools.http_server_connector import HttpServerConnector
from init_helpers.dict_to_dataclass import dict_to_dataclass


logger = logging.getLogger(__name__)


@dataclass
class FDescriptor:
    id: str
    size: int
    full_file_name: str
    file_extension: str
    content_type: str
    quoted_full_file_name: str
    full_file_name: str


@dataclass
class FileDescriptor(FDescriptor):
    thumbnail: Optional[FDescriptor]


class FileServerConnector:

    Config = HttpServerConnector.Config
    Context = HttpServerConnector.Context

    def __init__(self, config: Config, context: Context):
        self._connector = HttpServerConnector(config, context)

    async def get_file(self, file_id: str, range_: Optional[str] = None) -> bytes:
        args = {"id": file_id}
        headers = self._construct_headers()
        if range_ is not None:
            headers["Range"] = range_

        return await self._connector.get("/file/get", args, headers=headers)

    async def get_file_info(self, file_id: str) -> FileDescriptor:
        args = {"id": file_id}
        answer = await self._connector.get("/file/info", args, headers=self._construct_headers())
        return dict_to_dataclass(answer["result"], FileDescriptor)

    async def add_file(self, file: bytes, filename: str, is_create_thumbnail: bool = False) -> FileDescriptor:
        args = {"filename": filename, "is_create_thumbnail": 'true' if is_create_thumbnail else 'false'}
        answer = await self._connector.post("/file/add", payload=file, url_query=args,
                                            headers=self._construct_headers())
        return dict_to_dataclass(answer["result"], FileDescriptor)

    async def delete_file(self, file_id: str) -> FileDescriptor:
        args = {"id": file_id}
        answer = await self._connector.post("/file/delete", payload=None,
                                            url_query=args, headers=self._construct_headers())
        return dict_to_dataclass(answer["result"], FileDescriptor)

    def _construct_headers(self) -> dict[str, str]:
        headers = {"Connection": "Keep-Alive"}
        return headers
