#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>

from abc import ABC, abstractmethod
from dataclasses import dataclass
from typing import Any, Literal, Awaitable

from prometheus_tools.metrics_registry import MetricsRegistry


class AbstractFileStorage(ABC):
    @dataclass
    class Context:
        metrics: MetricsRegistry

    def __init__(self, context: Context) -> None:
        self.context = context
        self.class_name = type(self).__name__

    async def save(self, content: bytes, relative_path: str, name: str, allow_rewrite: bool = False) -> None:
        return await self._wrap_execution('save', self._save(content, relative_path, name, allow_rewrite))

    async def load(self, relative_path: str, name: str, offset: int = 0, size: int = -1) -> bytes:
        return await self._wrap_execution('load', self._load(relative_path, name, offset, size))

    async def delete(self, relative_path: str, name: str) -> None:
        return await self._wrap_execution('delete', self._delete(relative_path, name))

    async def check_file_existence(self, relative_path: str, name: str) -> None:
        return await self._wrap_execution('check_file_existence', self._check_file_existence(relative_path, name))

    async def _wrap_execution(
            self, action: Literal['save', 'load', 'delete', 'check_file_existence'], handler: Awaitable[Any]
    ) -> Any:
        with self.context.metrics.track(
                'file_storage__handlers', {'storage': self.class_name, 'action': action}, except_labels={'error': None}
        ) as tracker:
            try:
                return await handler
            except Exception as er:
                tracker.labels['error'] = type(er).__name__
                raise

    @abstractmethod
    async def _save(self, content: bytes, relative_path: str, name: str, allow_rewrite: bool = False) -> None:
        pass

    @abstractmethod
    async def _load(self, relative_path: str, name: str, offset: int = 0, size: int = -1) -> bytes:
        pass

    @abstractmethod
    async def _delete(self, relative_path: str, name: str) -> None:
        pass

    @abstractmethod
    async def _check_file_existence(self, relative_path: str, name: str) -> None:
        pass
