#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>

from pathlib import Path
from filetype import filetype
from typing import Optional

from .extension_to_mime_type import extension_to_mime_type


def define_file_extension_by_file_name(file_name: str) -> Optional[str]:
    if (file_extension := Path(file_name).suffix.lower()) and file_extension in extension_to_mime_type:
        return file_extension
    return None


def define_file_extension_by_content(file_content: bytes) -> Optional[str]:
    if file_extension := filetype.guess_extension(file_content):
        return f".{file_extension}"
    return None


def define_file_mime_type_by_content(file_content: bytes) -> Optional[str]:
    file_mime_type = filetype.guess_mime(file_content)
    return file_mime_type


def define_file_mime_type_by_extension(file_extension: str) -> str:
    return extension_to_mime_type.get(file_extension, "application/octet-stream")


def define_file_extension(file_name: str, file_content: bytes) -> Optional[str]:
    if file_extension_from_file_name := define_file_extension_by_file_name(file_name):
        return file_extension_from_file_name
    return define_file_extension_by_content(file_content)


def define_file_mime_type(file_name: str, file_content: bytes) -> Optional[str]:
    if file_extension_from_file_name := define_file_extension_by_file_name(file_name):
        return define_file_mime_type_by_extension(file_extension_from_file_name)
    return define_file_mime_type_by_content(file_content)
