#  Copyright (C) 2023
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasiliev Ivan <i.vasiliev@technokert.ru>
import enum
from typing import Type

from .clickhouse_types import BigInteger, LongFloat, UnsignedBigInteger, Float, Integer, ClickHouseType
from .column import Column
from .filter_by import Condition
from ..utils.selectable import Selectable


class AggregationFunctionType(enum.StrEnum):
    avg = "avg"
    sum = "sum"
    min = "min"
    max = "max"
    count = "count"


class AggregationFunction(Selectable):

    def __init__(self,
                 function_type: AggregationFunctionType,
                 attribute: Column,
                 condition: Condition | None = None
                 ):
        self._function_type = function_type
        self._column = attribute
        self._condition = condition
        super(AggregationFunction, self).__init__(self.get_clickhouse_type(), nullable=False, name=self.__str__())

    def get_clickhouse_type(self) -> Type[ClickHouseType]:
        if self._function_type in {AggregationFunctionType.min, AggregationFunctionType.max}:
            return self._column.clickhouse_type
        elif self._function_type == AggregationFunctionType.avg:
            return LongFloat
        elif self._function_type == AggregationFunctionType.count:
            return UnsignedBigInteger
        elif self._function_type == AggregationFunctionType.sum:
            if issubclass(self._column.clickhouse_type, Float):
                return LongFloat
            elif issubclass(self._column.clickhouse_type, Integer):
                return BigInteger

        raise NotImplementedError()

    def to_selector(self) -> str:
        if self._condition:
            return f"{self._function_type}If({self._column.name}, {self._condition}) as {self.name}"

        return f"{self._function_type}({self._column.name}) as {self.name}"

    def __str__(self):
        return f"{self._column.name}_{self._function_type}"
