#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>

import logging
from dataclasses import dataclass
from pathlib import PosixPath

from async_tools import AsyncInitable

from ..abstract_file_storage import AbstractFileStorage
from ..exceptions import BucketAlreadyExist
from ..s3_file_storage.s3_interface import S3Interface, Range

logger = logging.getLogger(__name__)


class S3FileStorage(AbstractFileStorage, AsyncInitable):
    @dataclass
    class Config(S3Interface.Config):
        bucket_name: str = "abm-storage"
        root_dir: str | None = None

    Context = AbstractFileStorage.Context

    def __init__(self, config: Config, context: Context):
        AsyncInitable.__init__(self)
        AbstractFileStorage.__init__(self, context)
        logger.info(f"{type(self).__name__} init. Config: {config}")
        self._s3_interface = S3Interface(config)
        self._bucket_name = config.bucket_name
        self._root_dir = (config.root_dir or "").lstrip("/")

    async def _async_init(self) -> None:
        try:
            await self._s3_interface.create_bucket(self._bucket_name)
        except BucketAlreadyExist:
            logger.info(f"Bucket already exist. bucket_name = {self._bucket_name}")
        else:
            logger.info(f"Bucket created. bucket_name = {self._bucket_name}")

    async def _save(self, content: bytes, relative_path: str, name: str, allow_rewrite: bool = False) -> None:
        await self._s3_interface.upload_file(bucket_name=self._bucket_name,
                                             payload=content,
                                             key=self._create_absolute_path(relative_path, name),
                                             allow_rewrite=allow_rewrite,
                                             )

    async def _load(self, relative_path: str, name: str, offset: int = 0, size: int = -1) -> bytes:
        return await self._s3_interface.get_file(bucket_name=self._bucket_name,
                                                 key=self._create_absolute_path(relative_path, name),
                                                 range_=Range(offset, size),
                                                 )

    async def _delete(self, relative_path: str, name: str) -> None:
        return await self._s3_interface.delete_file(bucket_name=self._bucket_name,
                                                    keys=self._create_absolute_path(relative_path, name),
                                                    )

    async def _check_file_existence(self, relative_path: str, name: str) -> bool:
        return await self._s3_interface.check_file_existence(bucket_name=self._bucket_name,
                                                             key=self._create_absolute_path(relative_path, name),
                                                             )

    def _create_absolute_path(self, relative_path: str, name: str) -> str:
        return str(PosixPath(self._root_dir, relative_path, name))
